package internal

import (
	"bytes"
	"fmt"
	"net/http"
	"os"
	"time"

	"github.com/labstack/echo/v4"
	"github.com/shirou/gopsutil/v3/process"

	"a.yandex-team.ru/library/go/core/metrics/solomon"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/handlers"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/knifeunittest"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmcache"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmtypes"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

const (
	sleepTime = 60
)

type Tvm struct {
	httpClient     *http.Client
	config         *tvmtypes.OptimizedConfig
	stop           chan bool
	verbose        bool
	cacheDirectory string
	state          handlers.StateHolder
	solomonReg     *solomon.Registry
}

// NewTvm creates Tvm and validates config
func NewTvm(
	httpClient *http.Client,
	config *tvmtypes.Config,
	verbose bool,
	cacheDir string,
) *Tvm {
	if verbose {
		logger.Log().Debugf("NewTvm(...) is called")
		if cacheDir != "" {
			logger.Log().Infof("Cache is enabled")
		}
	}

	if verbose {
		logConfiguration(config)
	}

	tvm := &Tvm{
		httpClient:     httpClient,
		config:         tvmtypes.NewOptimizedConfig(config),
		verbose:        verbose,
		cacheDirectory: cacheDir,
	}

	solomonOpts := solomon.NewRegistryOpts()
	tvm.solomonReg = solomon.NewRegistry(solomonOpts)

	return tvm
}

func logConfiguration(config *tvmtypes.Config) {
	logger.Log().Debugf("Blackbox env: %d", config.BbEnvType)
	logger.Log().Debugf("Clients:")
	var buffer bytes.Buffer
	for k, v := range config.Clients {
		buffer.WriteString(fmt.Sprintf("Alias: %s, TVMID: %d, Destinations: ", k, v.SelfTvmID))
		var dstBuffer bytes.Buffer
		for client, tvmid := range v.Dsts {
			dstBuffer.WriteString(fmt.Sprintf("Dst: (Alias: %s, TVMID: %d), ", client, tvmid))
		}

		str := dstBuffer.String()
		buffer.WriteString(str[:] + "\n")
	}
	logger.Log().Debugf(buffer.String())
}

func (tvm *Tvm) Start(useUnittest bool, unittestRolesDir string) error {
	if tvm.verbose {
		logger.Log().Infof("Starting...")
		if useUnittest {
			logger.Log().Infof("UnittestMode is enabled")
		}
	}

	if useUnittest {
		u, err := knifeunittest.NewUnittestState(tvm.config, unittestRolesDir)
		if err != nil {
			logger.Log().Errorf("%s", err)
			return err
		}
		tvm.state = u
		return nil
	}

	cachedState, err := createCache(tvm)
	if err != nil {
		return err
	}
	tvm.state = cachedState

	tvm.stop = make(chan bool)
	go func() {
		heartbeat := time.NewTicker(sleepTime * time.Second)

		for {
			select {
			case <-tvm.stop:
				logger.Log().Infof("Quitting update goroutine...")
				return

			case <-heartbeat.C:
				if tvm.verbose {
					logger.Log().Debugf("Starting tvmcache.Update()...")
				}
				if err := tvm.state.Update(tvm.config, tvm.httpClient); err != nil {
					logger.Log().Warnf("tvmcache.Update() error: %s", err)
				}
			}
		}
	}()

	return nil
}

func createCache(tvm *Tvm) (*tvmcache.Cache, error) {
	cachedState, err := tvmcache.NewCache(tvm.config.GetBbEnv(), tvm.cacheDirectory)
	if err != nil {
		logger.Log().Errorf("%s", err)
		return nil, err
	}

	if err := cachedState.FetchFromDisk(tvm.config); err != nil {
		logger.Log().Warnf("Failed to read cache from disk: %s", err)
	} else {
		return cachedState, nil
	}

	if err := cachedState.Update(tvm.config, tvm.httpClient); err != nil {
		return nil, err
	}

	return cachedState, nil
}

func (tvm *Tvm) Stop() {
	logger.Log().Infof("Stopping...")
	if tvm.stop == nil {
		logger.Log().Infof("Nothing to stop")
	} else {
		close(tvm.stop)
		logger.Log().Infof("Stopped")
	}
}

// GetTicketHandler creates HTTP handler for '/tvm/tickets'
func (tvm *Tvm) GetTicketHandler() echo.HandlerFunc {
	ticketHandler := handlers.NewTicketHandler(tvm.config)
	return ticketHandler.GetTicketHandler(tvm.state)
}

// GetTicketHandlerV2 creates HTTP handler for '/v2/tickets'
func (tvm *Tvm) GetTicketHandlerV2() echo.HandlerFunc {
	return handlers.GetTicketHandlerV2(tvm.config, tvm.state)
}

// GetPingHandler creates HTTP handler for '/tvm/ping'
func (tvm *Tvm) GetPingHandler() echo.HandlerFunc {
	return handlers.PingHandler(tvm.state)
}

// GetPingHandler creates HTTP handler for '/ping'
func (tvm *Tvm) GetPingHandlerV2() echo.HandlerFunc {
	return handlers.PingHandlerV2(tvm.state)
}

// GetKeysHandler creates HTTP handler for '/tvm/keys'
func (tvm *Tvm) GetKeysHandler() echo.HandlerFunc {
	return handlers.KeysHandler(tvm.state)
}

// GetCheckSrvTicketHandler creates HTTP handler for '/tvm/checksrv'
func (tvm *Tvm) GetCheckSrvTicketHandler() echo.HandlerFunc {
	ticketHandler := handlers.NewTicketHandler(tvm.config)
	return ticketHandler.GetCheckSrvTicketHandler(tvm.state)
}

// GetCheckUsrHandler creates HTTP handler for '/tvm/checkusr'
func (tvm *Tvm) GetCheckUsrHandler() echo.HandlerFunc {
	return handlers.GetCheckUsrHandler(tvm.state)
}

// GetMetaHandler creates HTTP handler for '/tvm/private_api/__meta__'
func (tvm *Tvm) GetMetaHandler() echo.HandlerFunc {
	return handlers.GetMetaHandler(tvm.config)
}

// GetForceUpdateHandler creates HTTP handler for '/tvm/cache/force_update'
func (tvm *Tvm) GetForceUpdateHandler() echo.HandlerFunc {
	return handlers.ForceUpdateHandler(tvm.state, tvm.config, tvm.httpClient)
}

func (tvm *Tvm) GetSolomonReg() *solomon.Registry {
	return tvm.solomonReg
}

// GetSolomonMetricsHandler creates HTTP handler for '/solomon/metrics'
func (tvm *Tvm) GetSolomonMetricsHandler() echo.HandlerFunc {
	selfProc, err := process.NewProcess(int32(os.Getpid()))
	if err != nil {
		logger.Log().Debugf("Failed to get process info: %s", err)
	}
	return handlers.NewSolomonMetricsHandler(tvm.solomonReg, selfProc)
}

func (tvm *Tvm) GetSolomonAuthMiddleware() echo.MiddlewareFunc {
	config := tvm.config.GetSolomonConfig()
	if config == nil {
		return func(next echo.HandlerFunc) echo.HandlerFunc {
			return func(ctx echo.Context) (err error) { return next(ctx) }
		}
	}

	return handlers.NewSolomonTVMAuthMiddleware(config, tvm.state)
}

// GetRolesV2 creates HTTP handler for '/v2/roles'
func (tvm *Tvm) GetRolesV2() echo.HandlerFunc {
	return handlers.GetRolesHandlerV2(tvm.config, tvm.state)
}

// GetCheckV2 creates HTTP handler for '/v2/check'
func (tvm *Tvm) GetCheckV2() echo.HandlerFunc {
	return handlers.CheckHandlerV2(tvm.config, tvm.state)
}
