package knifeunittest

import (
	"fmt"
	"net/http"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/crypto"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmcache"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmcontext"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmtypes"
)

const (
	unittestSecret = "fake_secret"
)

// UnittestState stores state for handlers
type UnittestState struct {
	config     *tvmtypes.OptimizedConfig
	serviceCtx *tvmcontext.ServiceContext
	userCtx    *tvmcontext.UserContext
	privateKey *crypto.RWPrivateKey
	roles      unittestRoles
}

// NewUnittestState create state
func NewUnittestState(config *tvmtypes.OptimizedConfig, unittestRolesDir string) (*UnittestState, error) {
	requiredSlugs := make([]string, 0)
	for _, v := range config.GetConfig().Clients {
		if v.Secret != unittestSecret && v.Secret != "" {
			return nil, fmt.Errorf("all secrets in config must be equal to '%s' or be empty", unittestSecret)
		}
		if v.IdmSlug != "" {
			if unittestRolesDir == "" {
				return nil, fmt.Errorf("dir with roles for unittest mode is not configured")
			}
			requiredSlugs = append(requiredSlugs, v.IdmSlug)
		}
	}

	var roles unittestRoles
	if len(requiredSlugs) != 0 {
		var err error
		roles, err = readRolesFromDir(unittestRolesDir, requiredSlugs)
		if err != nil {
			return nil, err
		}
	}

	u, err := tvmcontext.NewUserContext(publicKeys, config.GetBbEnv())
	if err != nil {
		return nil, err
	}
	s, err := tvmcontext.NewServiceContext(publicKeys)
	if err != nil {
		return nil, err
	}
	p, err := crypto.RWPrivateKeyFromString(crypto.TvmKnifePrivateKey)
	if err != nil {
		return nil, err
	}

	return &UnittestState{
		config:     config,
		serviceCtx: s,
		userCtx:    u,
		privateKey: p,
		roles:      roles,
	}, nil
}

// GetDiag creates diagnostics for /tvm/ping
func (c *UnittestState) GetDiag() tvmcache.Diag {
	d := tvmcache.DiagState{
		Status:     tvmcache.StatusOk,
		LastUpdate: time.Now(),
		LastErr:    nil,
	}

	return tvmcache.Diag{
		TicketErrors: make(map[tvmcache.ServiceTicketKey]string),
		Tickets:      d,
		Keys:         d,
	}
}

func (c *UnittestState) GetRoles(slug string) (*tvm.Roles, error) {
	if c.roles == nil {
		return nil, xerrors.Errorf("Roles were not configured at all")
	}

	res, found := c.roles[slug]
	if !found {
		return nil, xerrors.Errorf("slug '%s' was not configured", slug)
	}

	return res, nil
}

// Update is dummy
func (c *UnittestState) Update(config *tvmtypes.OptimizedConfig, client *http.Client) error {
	return nil
}

// ForceUpdate is dummy
func (c *UnittestState) ForceUpdate(config *tvmtypes.OptimizedConfig, client *http.Client) error {
	return nil
}
