package knifeunittest

import (
	"encoding/base64"
	"fmt"
	"time"

	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/library/cpp/tvmauth/src/protos"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/crypto"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmtypes"
)

// GetTicketUpdateTime returns now
func (c *UnittestState) GetTicketUpdateTime() time.Time {
	return time.Now()
}

// GetTicket returns new ticket
func (c *UnittestState) GetTicket(src tvm.ClientID, dst tvm.ClientID) (tvmtypes.Ticket, string, error) {
	er := checkSrcDst(c.config, src, dst)
	if len(er) > 0 {
		return "", er, nil
	}

	ticket, err := newServiceTicket(c.privateKey, src, dst)
	if err != nil {
		return "", "", err
	}

	return tvmtypes.Ticket(ticket), "", nil
}

func checkSrcDst(config *tvmtypes.OptimizedConfig, src tvm.ClientID, dst tvm.ClientID) string {
	s := config.FindClientByID(src)
	if s == nil {
		return fmt.Sprintf("Src %d was not configured", src)
	}

	found := false
	for _, v := range s.Dsts {
		if dst == v.ID {
			found = true
			break
		}
	}
	if !found {
		return fmt.Sprintf("Dst %d was not configured for src %d", dst, src)
	}

	return ""
}

func newServiceTicket(priv *crypto.RWPrivateKey, src tvm.ClientID, dst tvm.ClientID) (string, error) {
	p := &protos.Ticket{}

	keyInt := uint32(16)
	p.KeyId = &keyInt
	expirationTime := time.Now().Unix() + 2*3600
	p.ExpirationTime = &expirationTime

	p.Service = &protos.ServiceTicket{}
	srcInt := uint32(src)
	p.Service.SrcClientId = &srcInt
	dstInt := uint32(dst)
	p.Service.DstClientId = &dstInt

	pbytes, err := proto.Marshal(p)
	if err != nil {
		return "", fmt.Errorf("failed to serialize protobuf: %s", err.Error())
	}

	ticket := fmt.Sprintf("3:serv:%s:", base64.RawURLEncoding.EncodeToString(pbytes))
	sign, err := priv.Sign([]byte(ticket))
	if err != nil {
		return "", fmt.Errorf("failed to create signature: %s", err.Error())
	}

	return ticket + base64.RawURLEncoding.EncodeToString(sign), nil
}
