package middleware

import (
	"net/http"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/errs"
)

type ErrorResponse struct {
	Status string `json:"status"`
	Error  string `json:"error"`
}

const (
	StatusBadRequest            = "BAD_REQUEST"
	StatusInvalidParams         = "INVALID_PARAMS"
	StatusInvalidLocalAuthToken = "INVALID_LOCAL_AUTHTOKEN"
	StatusUnexpectedError       = "UNEXPECTED_ERROR"
)

func NewErrorHandleMiddlewareV2() echo.MiddlewareFunc {
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(ctx echo.Context) error {
			err := next(ctx)

			switch err := err.(type) {
			case nil:
				return nil

			case *echo.HTTPError:
				return ctx.JSON(err.Code, ErrorResponse{
					Status: StatusBadRequest,
					Error:  err.Error(),
				})

			case *errs.InvalidParam:
				return ctx.JSON(http.StatusBadRequest, ErrorResponse{
					Status: StatusInvalidParams,
					Error:  err.Error(),
				})

			case *errs.Unauthorized:
				return ctx.JSON(http.StatusUnauthorized, ErrorResponse{
					Status: StatusInvalidLocalAuthToken,
					Error:  err.Error(),
				})

			case *errs.Temporary:
				return ctx.JSON(http.StatusInternalServerError, ErrorResponse{
					Status: StatusUnexpectedError,
					Error:  err.Error(),
				})
			}

			return ctx.JSON(http.StatusInternalServerError, ErrorResponse{
				Status: StatusUnexpectedError,
				Error:  "Unhandled error: " + err.Error(),
			})
		}
	}
}
