package middleware

import (
	"crypto/subtle"
	"strings"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/core/metrics/solomon"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/errs"
)

const (
	unauthorizedMessage = "Invalid authentication token"
	authorizationHeader = "Authorization"
)

func NewAuthMiddleware(token string) echo.MiddlewareFunc {
	token = strings.ToLower(token)

	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) (err error) {
			authHeader := strings.ToLower(c.Request().Header.Get(authorizationHeader))

			if subtle.ConstantTimeCompare([]byte(authHeader), []byte(token)) != 1 {
				return &errs.Unauthorized{Message: unauthorizedMessage}
			}

			return next(c)
		}
	}
}

func NewServerVersionMiddleware(serverVersion string) echo.MiddlewareFunc {
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) (err error) {
			c.Response().Header().Set("Server", serverVersion)
			return next(c)
		}
	}
}

func NewSolomonHandlerMetricsMiddleware(routes []*echo.Route, registry *solomon.Registry) echo.MiddlewareFunc {
	paths := make(map[string]metrics.Counter)

	for _, r := range routes {
		if _, exists := paths[r.Path]; exists {
			continue
		}

		rps := registry.WithTags(map[string]string{"path": r.Path}).Counter("http.requests")
		// Convert counters to rated rps counters
		solomon.Rated(rps)
		paths[r.Path] = rps
	}

	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) (err error) {
			p, ok := paths[c.Request().URL.Path]
			if ok {
				p.Inc()
			}
			return next(c)
		}
	}
}
