package tirole

import (
	"fmt"
	"net/http"
	"net/url"
	"time"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmversion"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type Tirole struct {
	client *resty.Client
}

func NewTirole(tiroleURL *url.URL, client *http.Client) *Tirole {
	httpc := resty.NewWithClient(client).
		SetBaseURL(tiroleURL.String()).
		SetTimeout(10 * time.Second).
		SetRedirectPolicy(resty.NoRedirectPolicy())

	return &Tirole{
		client: httpc,
	}
}

func (t *Tirole) GetRoles(slug, serviceTicket, currentRevision string) (*tvm.Roles, error) {
	req := t.buildRequest(slug, serviceTicket, currentRevision)

	resp, err := req.Get("/v1/get_actual_roles")
	if err != nil {
		logger.Log().Warnf("tirole.GetRawRoles(). %s", err)
		return nil, err
	}
	logger.Log().Debugf(
		"tirole.GetRawRoles(). Succeed to perform request for roles to %s (request_id=%s). code=%d",
		t.client.BaseURL,
		resp.Header().Get("X-Request-Id"),
		resp.StatusCode(),
	)

	blob, err := procResponse(resp, slug)
	if err != nil || blob == nil {
		return nil, err
	}

	return tvm.NewRoles(blob)
}

func (t *Tirole) buildRequest(slug, serviceTicket, currentRevision string) *resty.Request {
	req := t.client.R().
		SetQueryParams(map[string]string{
			"system_slug": slug,
			"lib_version": tvmversion.GetVersion(),
		}).
		SetHeader("X-Ya-Service-Ticket", serviceTicket)
	if currentRevision != "" {
		req.SetHeader("If-None-Match", fmt.Sprintf(`"%s"`, currentRevision))
	}

	return req
}

func procResponse(resp *resty.Response, slug string) ([]byte, error) {
	if resp.StatusCode() == http.StatusNotModified {
		logger.Log().Debugf("tirole.GetRawRoles(). Roles were not modified fot slug='%s'", slug)
		return nil, nil
	}
	if resp.StatusCode() != http.StatusOK {
		logger.Log().Warnf("tirole.GetRawRoles() bad status code. Code %d, Body: %s", resp.StatusCode(), resp.Body())
		return nil, fmt.Errorf("HTTP code %d: %s", resp.StatusCode(), resp.Body())
	}

	codecHeader := resp.Header().Get("X-Tirole-Compression")
	blob, err := decode(resp.Body(), codecHeader)
	if err != nil {
		logger.Log().Warnf("tirole.GetRawRoles(). %s", err)
		return nil, err
	}

	return blob, nil
}
