package tirole

import (
	"bytes"
	"compress/gzip"
	"io/ioutil"

	"github.com/andybalholm/brotli"
	"github.com/klauspost/compress/zstd"

	"a.yandex-team.ru/library/go/core/xerrors"
)

type codec interface {
	Decode([]byte) ([]byte, error)
}

func getCodec(codec string) (codec, error) {
	switch codec {
	case "brotli":
		return &brotliDecoder{}, nil
	case "gzip":
		return &gzipDecoder{}, nil
	case "zstd":
		return &zstdDecoder{}, nil
	default:
		return nil, xerrors.Errorf("unknown codec: '%s'", codec)
	}
}

type brotliDecoder struct{}

func (d brotliDecoder) Decode(blob []byte) ([]byte, error) {
	reader := brotli.NewReader(bytes.NewBuffer(blob))
	return ioutil.ReadAll(reader)
}

type gzipDecoder struct{}

func (d gzipDecoder) Decode(blob []byte) ([]byte, error) {
	reader, err := gzip.NewReader(bytes.NewBuffer(blob))
	if err != nil {
		return nil, err
	}

	return ioutil.ReadAll(reader)
}

type zstdDecoder struct{}

func (d zstdDecoder) Decode(blob []byte) ([]byte, error) {
	reader, err := zstd.NewReader(nil, zstd.WithDecoderConcurrency(1))
	if err != nil {
		return nil, err
	}
	defer reader.Close()

	dst := make([]byte, 0)
	return reader.DecodeAll(blob, dst)
}
