package tirole

import (
	"crypto/sha256"
	"encoding/hex"
	"strconv"
	"strings"

	"a.yandex-team.ru/library/go/core/xerrors"
)

func decode(blob []byte, codec string) ([]byte, error) {
	if codec == "" {
		return blob, nil
	}

	info, err := parseCodecInfo(codec)
	if err != nil {
		return nil, xerrors.Errorf("failed to parse codec info from tirole: '%s'. %v", codec, err)
	}

	dec, err := getCodec(info.Type)
	if err != nil {
		return nil, xerrors.Errorf("failed to get codec for roles from tirole: '%s'. %v", codec, err)
	}

	result, err := dec.Decode(blob)
	if err != nil {
		return nil, xerrors.Errorf("failed to decode roles from tirole: '%s'. %v", codec, err)
	}

	if err := verifyDecodedResult(result, info); err != nil {
		return nil, err
	}

	return result, nil
}

type codecInfo struct {
	Type   string
	Size   uint64
	Sha256 string
}

func parseCodecInfo(codec string) (*codecInfo, error) {
	tokens := strings.Split(codec, ":")
	if version := tokens[0]; version != "1" {
		return nil, xerrors.Errorf("unknown codec format version; known: 1")
	}
	if len(tokens) != 4 {
		return nil, xerrors.Errorf("malformed codec")
	}

	codecType := tokens[1]

	decodedSize := tokens[2]
	size, err := strconv.ParseUint(decodedSize, 10, 64)
	if err != nil {
		return nil, xerrors.Errorf("decoded blob size is not number")
	}

	sha := tokens[3]
	expectedSha256Size := 2 * 32
	if len(sha) != expectedSha256Size {
		return nil, xerrors.Errorf(
			"sha256 of decoded blob has invalid length: expected %d, got %d",
			expectedSha256Size,
			len(sha),
		)
	}

	return &codecInfo{
		Type:   codecType,
		Size:   size,
		Sha256: sha,
	}, nil
}

func verifyDecodedResult(result []byte, info *codecInfo) error {
	if err := verifySize(result, info.Size); err != nil {
		return err
	}
	if err := verifySha256(result, info.Sha256); err != nil {
		return err
	}

	return nil
}

func verifySize(result []byte, expectedSize uint64) error {
	if uint64(len(result)) != expectedSize {
		return xerrors.Errorf("decoded size is unexpected: actual=%d vs expected=%d", len(result), expectedSize)
	}

	return nil
}

func verifySha256(result []byte, expectedSha256 string) error {
	sh := sha256.New()
	if _, err := sh.Write(result); err != nil {
		return xerrors.Errorf("failed to calculate sha256 for decoded roles: %v", err)
	}
	hash := hex.EncodeToString(sh.Sum(nil))

	hash = strings.ToLower(hash)
	expectedSha256 = strings.ToLower(expectedSha256)

	if hash != expectedSha256 {
		return xerrors.Errorf("decoded blob has bad sha256: expected=%s, actual=%s", expectedSha256, hash)
	}

	return nil
}
