package tvmapi

import (
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/url"
	"strconv"
	"time"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmtypes"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmversion"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

const (
	unknownError = "unknown error"
)

type TvmAPI struct {
	client *resty.Client
}

var errorInvalidJSONTickets = errors.New("got invalid json from tvm-api (tickets)")

func NewTvmAPI(tvmURL *url.URL, client *http.Client) *TvmAPI {
	httpc := resty.NewWithClient(client).
		SetBaseURL(tvmURL.String()).
		SetTimeout(10 * time.Second).
		SetRedirectPolicy(resty.NoRedirectPolicy())

	return &TvmAPI{
		client: httpc,
	}
}

func (t *TvmAPI) GetKeys() (string, error) {
	req := t.client.R().
		SetQueryParam("lib_version", tvmversion.GetVersion())

	resp, err := req.Get("/2/keys")
	if err != nil {
		logger.Log().Warnf("tvmapi.GetKeys(). %s", err)
		return "", err
	}

	if resp.StatusCode() != http.StatusOK {
		logger.Log().Warnf("tvmapi.GetKeys() bad status code. Code %d, Body: %s", resp.StatusCode(), resp.Body())
		return "", fmt.Errorf("HTTP code %d: %s", resp.StatusCode(), resp.Body())
	}

	return string(resp.Body()), nil
}

func buildTicketsRequestBody(secret string, src tvm.ClientID, dsts []tvmtypes.Dst) (string, string, error) {
	ts := strconv.FormatInt(time.Now().Unix(), 10)
	srcStr := strconv.FormatUint(uint64(src), 10)
	dstStr := ""
	for i, num := range dsts {
		if i != 0 {
			dstStr += ","
		}
		dstStr += strconv.FormatUint(uint64(num.ID), 10)
	}
	sign, err := signRequest(ts, dstStr, secret)
	if err != nil {
		return "", "", xerrors.Errorf("sign request failed: %v", err)
	}

	post := fmt.Sprintf(
		"grant_type=client_credentials&ts=%s&src=%s&dst=%s&sign=%s&lib_version=%s",
		ts, srcStr, dstStr, sign, tvmversion.GetVersion(),
	)
	logable := fmt.Sprintf(
		"grant_type=client_credentials&ts=%s&src=%s&dst=%s",
		ts, srcStr, dstStr)

	return post, logable, nil
}

func (t *TvmAPI) GetTickets(secret string, src tvm.ClientID, dsts []tvmtypes.Dst) (tvmtypes.TicketsInfo, error) {
	post, logable, err := buildTicketsRequestBody(secret, src, dsts)
	if err != nil {
		logger.Log().Errorf("failed to build /2/tickets request: %s", err)
		return tvmtypes.TicketsInfo{}, err
	}
	logger.Log().Infof("tvmapi.GetTickets(). Sending request for %s", logable)

	req := t.client.R().
		SetBody(post)
	resp, err := req.Post("/2/ticket")
	if err != nil {
		logger.Log().Warnf("tvmapi.GetTickets(). Request error: %s", err)
		return tvmtypes.TicketsInfo{}, err
	}
	if resp.StatusCode() != http.StatusOK {
		logger.Log().Warnf("tvmapi.GetTickets(). Bad status code: %d, Body: %s", resp.StatusCode(), resp.Body())
		return tvmtypes.TicketsInfo{}, fmt.Errorf("HTTP code %d: %s : %s", resp.StatusCode(), post, resp.Body())
	}

	return parseResponse(resp.Body())
}

func parseResponse(body []byte) (tvmtypes.TicketsInfo, error) {
	var personMap map[string]map[string]string
	err := json.Unmarshal(body, &personMap)
	if err != nil {
		logger.Log().Errorf("tvmapi.GetTickets(). Json unmarshal error: %s", err)
		return tvmtypes.TicketsInfo{}, errorInvalidJSONTickets
	}

	res := tvmtypes.TicketsInfo{
		Tickets: make(map[tvm.ClientID]tvmtypes.Ticket),
		Errors:  make(map[tvm.ClientID]string),
	}
	for dst, v := range personMap {
		i, err := strconv.ParseUint(dst, 10, 32)
		if err != nil {
			return tvmtypes.TicketsInfo{}, fmt.Errorf("tvmapi.GetTickets(). dst is not number: %s", dst)
		}

		str, ok := v["ticket"]
		if ok {
			res.Tickets[tvm.ClientID(i)] = tvmtypes.Ticket(str)
			continue
		}

		str, errOk := v["error"]
		if errOk {
			logger.Log().Warnf("tvmapi.GetTickets(). Failed to get ticket for %s", fmt.Sprintf("%s: %s", dst, str))
		} else {
			logger.Log().Warnf("tvmapi.GetTickets(). Failed to get ticket %s", body)
			str = unknownError
		}
		res.Errors[tvm.ClientID(i)] = str
	}

	return res, nil
}
