package tvmcache

import (
	"errors"
	"fmt"
	"net/http"
	"os"
	"sync"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tirole"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmapi"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmcontext"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmtypes"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

const (
	attemptMinInterval = 600
)

type Cache struct {
	keysCache       *tvmKeysCache
	ticketCache     *tvmTicketCache
	rolesCache      *rolesCache
	cacheDirectory  string
	lastForceUpdate time.Time
	mtx             sync.RWMutex
}

func NewCache(env tvm.BlackboxEnv, dirname string) (*Cache, error) {
	if dirname != "" {
		if err := checkCacheDirectory(dirname); err != nil {
			return nil, err
		}
	}

	res := &Cache{
		keysCache:      NewTvmKeysCache(dirname, env),
		ticketCache:    NewTicketsCache(dirname),
		rolesCache:     NewRolesCache(dirname),
		cacheDirectory: dirname,
	}

	return res, nil
}

type Diag struct {
	TicketErrors map[ServiceTicketKey]string
	Tickets      DiagState
	Keys         DiagState
	Roles        DiagState
}

func checkCacheDirectory(dirname string) error {
	stat, err := os.Stat(dirname)
	if os.IsNotExist(err) {
		return xerrors.Errorf("failed to stat disk cache dir: %v", err)
	}

	if !stat.IsDir() {
		return errors.New("this is not a directory")
	}

	return nil
}

func (c *Cache) FetchFromDisk(config *tvmtypes.OptimizedConfig) error {
	if c.cacheDirectory == "" {
		return errors.New("disk cache is disabled")
	}

	errs := []error{
		c.keysCache.FetchFromDisk(),
		c.ticketCache.FetchFromDisk(config),
		c.rolesCache.FetchFromDisk(config),
	}

	return errorArrayToSingle(errs)
}

func (c *Cache) Update(config *tvmtypes.OptimizedConfig, client *http.Client) error {
	now := time.Now()
	isTicketsAttempt := c.ticketCache.IsTimeForAttempt(now)
	isKeysAttempt := c.keysCache.IsTimeForAttempt(now)
	isRolesAttempt := c.rolesCache.IsTimeForAttempt(config.GetClients(), now)

	if !isTicketsAttempt && !isKeysAttempt && !isRolesAttempt {
		return nil
	}

	c.mtx.Lock()
	defer c.mtx.Unlock()
	logger.Log().Infof("Updating started")

	tvmAPIClient := tvmapi.NewTvmAPI(config.GetConfig().Backends.TvmURL, client)

	errs := make([]error, 0)
	if isKeysAttempt {
		if err := c.keysCache.Update(tvmAPIClient); err != nil {
			errs = append(errs, err)
		}
	}

	if isTicketsAttempt {
		srvctx, err := c.GetServiceContext()
		if err != nil {
			return errors.New("failed to check tickets after fetching: no keys")
		}

		if err := c.ticketCache.Update(tvmAPIClient, srvctx, config.GetClients()); err != nil {
			errs = append(errs, err)
		}
	}

	if isRolesAttempt {
		tiroleClient := tirole.NewTirole(config.GetConfig().Backends.TiroleURL, client)
		if err := c.rolesCache.Update(
			c.newTiroleGate(tiroleClient, config),
			config.GetClients(),
		); err != nil {
			errs = append(errs, err)
		}
	}

	logger.Log().Infof("Updating finished")

	return errorArrayToSingle(errs)
}

func (c *Cache) getLastForceUpdateTime() time.Time {
	c.mtx.RLock()
	defer c.mtx.RUnlock()
	return c.lastForceUpdate
}

func (c *Cache) ForceUpdate(config *tvmtypes.OptimizedConfig, client *http.Client) error {
	now := time.Now()
	if allowedTime := c.getLastForceUpdateTime().Add(time.Minute); now.Before(allowedTime) {
		d := allowedTime.Sub(now)
		return fmt.Errorf("ForceUpdate() is expensive operation: you can run it in %d seconds", int(d.Seconds()))
	}

	c.mtx.Lock()
	defer c.mtx.Unlock()
	c.lastForceUpdate = now
	logger.Log().Infof("Force updating started")

	tvmAPIClient := tvmapi.NewTvmAPI(config.GetConfig().Backends.TvmURL, client)

	logger.Log().Infof("ForceUpdate(). updating keys...")
	err := c.keysCache.Update(tvmAPIClient)
	if err != nil {
		logger.Log().Warnf("ForceUpdate(). updating failed for keys: %s", err)
		return err
	}
	logger.Log().Infof("ForceUpdate(). updating keys... Succeed")

	logger.Log().Infof("ForceUpdate(). updating tickets...")
	srvctx, err := c.GetServiceContext()
	if err != nil {
		logger.Log().Warnf("ForceUpdate(). unreachable case: %s", err)
		return xerrors.Errorf("failed to check tickets after fetching: no keys: %v", err)
	}

	if err := c.ticketCache.Update(tvmAPIClient, srvctx, config.GetClients()); err != nil {
		logger.Log().Warnf("ForceUpdate(). updating failed for tickets: %s", err)
		return err
	}
	logger.Log().Infof("ForceUpdate(). updating tickets... Succeed")

	logger.Log().Infof("ForceUpdate(). updating roles...")
	tiroleClient := tirole.NewTirole(config.GetConfig().Backends.TiroleURL, client)
	if err := c.rolesCache.Update(
		c.newTiroleGate(tiroleClient, config),
		config.GetClients(),
	); err != nil {
		logger.Log().Warnf("ForceUpdate(). updating failed for roles: %s", err)
		return err
	}
	logger.Log().Infof("ForceUpdate(). updating roles... Succeed")

	logger.Log().Infof("Force updating finished")
	return nil
}

func (c *Cache) GetKeys() (string, time.Time, error) {
	return c.keysCache.GetKeys()
}

func (c *Cache) GetServiceContext() (*tvmcontext.ServiceContext, error) {
	return c.keysCache.GetServiceContext()
}

func (c *Cache) GetUserContext() (*tvmcontext.UserContext, error) {
	return c.keysCache.GetUserContext()
}

func (c *Cache) GetTicketUpdateTime() time.Time {
	return c.ticketCache.GetTicketUpdateTime()
}

func (c *Cache) GetTicket(src tvm.ClientID, dst tvm.ClientID) (tvmtypes.Ticket, string, error) {
	return c.ticketCache.GetTicket(src, dst)
}

func (c *Cache) GetRoles(slug string) (*tvm.Roles, error) {
	return c.rolesCache.GetRoles(slug)
}

func (c *Cache) GetDiag() Diag {
	now := time.Now()
	return Diag{
		TicketErrors: c.ticketCache.GetTicketErrors(),
		Tickets:      c.ticketCache.GetDiag(now),
		Keys:         c.keysCache.GetDiag(now),
		Roles:        c.rolesCache.GetDiag(now),
	}
}

func (c *Cache) newTiroleGate(tiroleClient *tirole.Tirole, config *tvmtypes.OptimizedConfig) tiroleGate {
	return &tiroleGateImpl{
		tvmapiClient: c.ticketCache,
		tiroleClient: tiroleClient,
		tiroleTvmID:  config.GetConfig().Backends.TiroleTvmID,
	}
}

func errorArrayToSingle(errs []error) error {
	var res error

	for _, err := range errs {
		if err == nil {
			continue
		}

		if res == nil {
			res = err
		} else {
			res = xerrors.Errorf("%v. %v", res, err)
		}
	}

	return res
}
