package tvmcache

import (
	"crypto/sha256"
	"crypto/subtle"
	"encoding/binary"
	"errors"
	"io/ioutil"
	"path"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmtypes"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

const (
	keysCacheFileName     = ".tvm-keys.cache"
	ticketsCacheFileName  = ".tvm-tickets.cache"
	cacheFilesPermissions = 0640
	minFileSize           = 48
)

func getCacheFileName(cacheDirectory, filename string) string {
	return path.Join(cacheDirectory, filename)
}

func checkHasSrcDst(tkts map[tvm.ClientID]tvmtypes.TicketsForOneClient, src tvm.ClientID, dst tvm.ClientID) bool {
	item, ok := tkts[src]
	if !ok {
		return false
	}

	if ticket, ok := item[dst]; ok && ticket != "" {
		return true
	}

	return false
}

func writeCacheFile(filename string, tmstamp uint64, data []byte) error {
	dataToWrite := make([]byte, 0)
	tmstampData := make([]byte, 8)
	binary.LittleEndian.PutUint64(tmstampData, tmstamp)

	sh := sha256.New()
	if _, err := sh.Write(tmstampData); err != nil {
		panic(err)
	}
	if _, err := sh.Write(data); err != nil {
		panic(err)
	}

	dataToWrite = append(dataToWrite, sh.Sum(nil)...)
	dataToWrite = append(dataToWrite, tmstampData...)
	dataToWrite = append(dataToWrite, data...)

	logger.Log().Debugf("Writing cache to file: %s", filename)
	err := ioutil.WriteFile(filename, dataToWrite, cacheFilesPermissions)
	if err == nil {
		logger.Log().Debugf("Writing cache to file succeed: %s", filename)
	} else {
		logger.Log().Errorf("Writing cache to file failed: %s. %s", filename, err)
	}
	return err
}

func readCacheFile(filename string) ([]byte, uint64, error) {
	data, err := ioutil.ReadFile(filename)
	if err != nil {
		return nil, 0, xerrors.Errorf("failed to read file: %v", err)
	}

	if len(data) < minFileSize {
		return nil, 0, errors.New("file is too small")
	}

	hashValue := data[0:32]
	timestampValue := data[32:40]
	resultData := data[40:]

	hashResult := sha256.Sum256(data[32:])

	if subtle.ConstantTimeCompare(hashResult[:], hashValue) != 1 {
		return nil, 0, errors.New("invalid cache data hash value")
	}

	return resultData, binary.LittleEndian.Uint64(timestampValue), nil
}
