package tvmcache

import (
	"errors"
	"time"

	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmcontext"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

const (
	keysUpdateInterval = 86400
)

type tvmKeysCache struct {
	bbenv          tvm.BlackboxEnv
	state          cacheState
	keys           string
	uctx           *tvmcontext.UserContext
	sctx           *tvmcontext.ServiceContext
	cacheDirectory string
}

func NewTvmKeysCache(cacheDirectory string, env tvm.BlackboxEnv) *tvmKeysCache {
	return &tvmKeysCache{
		bbenv:          env,
		cacheDirectory: cacheDirectory,
	}
}

func (c *tvmKeysCache) FetchFromDisk() error {
	filename := getCacheFileName(c.cacheDirectory, keysCacheFileName)
	keys, tmstamp, err := readCacheFile(filename)
	if err != nil {
		return err
	}

	if time.Now().Unix()-int64(tmstamp) > keysUpdateInterval {
		return errors.New("keys cache is too old, online updated is required")
	}

	err = c.setKeysFromString(string(keys), time.Unix(int64(tmstamp), 0))
	if err != nil {
		return err
	}
	logger.Log().Infof("Keys successfully read from file: %s", filename)

	return nil
}

func (c *tvmKeysCache) GetKeys() (string, time.Time, error) {
	c.state.mtx.RLock()
	defer c.state.mtx.RUnlock()

	if len(c.keys) == 0 {
		return "", time.Time{}, errors.New("tvm keys are empty")
	}
	return c.keys, c.state.lastUpdated, nil
}

func (c *tvmKeysCache) GetServiceContext() (*tvmcontext.ServiceContext, error) {
	c.state.mtx.RLock()
	defer c.state.mtx.RUnlock()

	if c.sctx == nil {
		return nil, errors.New("service context update fail")
	}

	return c.sctx, nil
}

func (c *tvmKeysCache) GetUserContext() (*tvmcontext.UserContext, error) {
	c.state.mtx.RLock()
	defer c.state.mtx.RUnlock()

	if c.uctx == nil {
		return nil, errors.New("user context update fail")
	}

	return c.uctx, nil
}

func (c *tvmKeysCache) GetDiag(now time.Time) DiagState {
	return c.state.GetDiag(now.Unix(), keysUpdateInterval)
}

func (c *tvmKeysCache) IsTimeForAttempt(now time.Time) bool {
	return c.state.IsTimeForAttempt(now.Unix(), keysUpdateInterval)
}

func (c *tvmKeysCache) setKeysFromString(keys string, updateTime time.Time) error {
	c.state.mtx.Lock()
	defer c.state.mtx.Unlock()

	c.keys = keys
	c.state.lastUpdated = updateTime
	c.state.lastError = nil

	sctx, err := tvmcontext.NewServiceContext(keys)
	if err != nil {
		c.state.lastError = err
		logger.Log().Errorf("Keys update error (can't create service context): %s", err)
		return err
	}
	c.sctx = sctx

	uctx, err := tvmcontext.NewUserContext(keys, c.bbenv)
	if err != nil {
		c.state.lastError = err
		logger.Log().Errorf("Keys update error (can't create user context): %s", err)
		return err
	}
	c.uctx = uctx

	return nil
}

func (c *tvmKeysCache) updateLastAttemptTime() {
	c.state.mtx.Lock()
	defer c.state.mtx.Unlock()

	c.state.lastAttempt = time.Now()
}

type tvmKeysGetter interface {
	GetKeys() (string, error)
}

func (c *tvmKeysCache) Update(tvmAPIClient tvmKeysGetter) error {
	logger.Log().Infof("Updating keys...")
	keys, err := tvmAPIClient.GetKeys()
	c.updateLastAttemptTime()

	if err != nil {
		logger.Log().Warnf("Keys update error %s", err)
		c.setLastError(err)
		return err
	}

	err = c.setKeysFromString(keys, time.Now())

	if err == nil {
		logger.Log().Infof("Updating keys... Succeed")

		if c.cacheDirectory != "" {
			filename := getCacheFileName(c.cacheDirectory, keysCacheFileName)
			_ = writeCacheFile(filename, uint64(time.Now().Unix()), []byte(keys))
		}
	}

	return err
}

func (c *tvmKeysCache) setLastError(err error) {
	c.state.mtx.Lock()
	defer c.state.mtx.Unlock()

	c.state.lastError = err
}
