package tvmcache

import (
	"fmt"
	"sync"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmtypes"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

const (
	rolesUpdateInterval = 600
)

type rolesCache struct {
	cacheDirectory  string
	mutex           sync.RWMutex
	bySlug          rolesStateBySlug
	updateForClient updateForClientFunc
}

type rolesState struct {
	state cacheState
	roles *tvm.Roles
}

type rolesStateBySlug map[string]*rolesState

type tiroleGate interface {
	GetServiceTicket(id tvm.ClientID) (string, error)
	CreateRolesReader(slug, serviceTicket string) tiroleRoleReader
}

type tiroleRoleReader func(currentRevision string) (*tvm.Roles, error)

type updateForClientFunc func(
	gate tiroleGate,
	client *tvmtypes.Client,
	currentRoles *rolesState,
	cacheDirectory string,
	now time.Time,
) (*rolesState, error)

func NewRolesCache(cacheDirectory string) *rolesCache {
	return &rolesCache{
		cacheDirectory:  cacheDirectory,
		bySlug:          make(rolesStateBySlug),
		updateForClient: updateForClient,
	}
}

func (c *rolesCache) FetchFromDisk(config *tvmtypes.OptimizedConfig) error {
	res, err := fetchFromDiskImpl(c.cacheDirectory, config)
	c.setRolesBySlug(res)
	return err
}

func fetchFromDiskImpl(cacheDirectory string, config *tvmtypes.OptimizedConfig) (rolesStateBySlug, error) {
	errs := make([]error, 0)
	res := make(rolesStateBySlug)

	for _, client := range config.GetClients() {
		if client.IdmSlug == "" {
			continue
		}

		filename := getCacheFileName(cacheDirectory, buildRolesFileName(client.IdmSlug))
		blob, tmstamp, err := readCacheFile(filename)
		if err != nil {
			err = xerrors.Errorf("fail to read file: %v", err)
			errs = append(errs, err)
			logger.Log().Debugf("roles.FetchFromDisk(). %s", err)
			continue
		}

		roles, err := tvm.NewRoles(blob)
		if err != nil {
			err = xerrors.Errorf("fail to parse roles from file: %s: %v", filename, err)
			errs = append(errs, err)
			logger.Log().Debugf("roles.FetchFromDisk(). %s", err)
			continue
		}

		res[client.IdmSlug] = &rolesState{
			state: cacheState{
				lastUpdated: time.Unix(int64(tmstamp), 0),
			},
			roles: roles,
		}
		logger.Log().Infof("Roles successfully read from file: %s", filename)
	}

	return res, errorArrayToSingle(errs)
}

func (c *rolesCache) IsTimeForAttempt(clients []*tvmtypes.Client, now time.Time) bool {
	roles := c.getRolesBySlug()

	for _, client := range clients {
		if client.IdmSlug == "" {
			continue
		}

		v, found := roles[client.IdmSlug]
		if !found {
			return true
		}

		if v.state.IsTimeForAttempt(now.Unix(), rolesUpdateInterval) {
			return true
		}
	}

	return false
}

func (c *rolesCache) GetDiag(now time.Time) DiagState {
	roles := c.getRolesBySlug()

	for _, v := range roles {
		diag := v.state.GetDiag(now.Unix(), rolesUpdateInterval)
		switch diag.Status {
		case StatusOk:
			continue
		case StatusWarning:
			return diag
		case StatusError:
			diag.Status = StatusWarning
			return diag
		}
	}

	return DiagState{}
}

func (c *rolesCache) GetRoles(slug string) (*tvm.Roles, error) {
	roles := c.getRolesBySlug()

	res, found := roles[slug]
	if !found {
		return nil, xerrors.Errorf("slug '%s' was not configured", slug)
	}

	return res.roles, nil
}

type rolesGetter interface {
	GetRoles(slug, serviceTicket, currentRevision string) (*tvm.Roles, error)
}

type serviceTicketGetter interface {
	GetTicket(src tvm.ClientID, dst tvm.ClientID) (tvmtypes.Ticket, string, error)
}

func (c *rolesCache) Update(gate tiroleGate, clients []*tvmtypes.Client) error {
	now := time.Now()
	currentRolesBySlug := c.getRolesBySlug()
	newRolesBySlug := make(rolesStateBySlug)
	errs := make([]error, 0)

	for _, client := range clients {
		if client.IdmSlug == "" {
			continue
		}
		currentRoles := currentRolesBySlug[client.IdmSlug]

		newRoles, err := c.updateForClient(gate, client, currentRoles, c.cacheDirectory, now)
		if err != nil {
			errs = append(errs, err)
		}
		if newRoles != nil {
			newRolesBySlug[client.IdmSlug] = newRoles
		}
	}

	c.setRolesBySlug(newRolesBySlug)

	return errorArrayToSingle(errs)
}

func updateForClient(
	gate tiroleGate,
	client *tvmtypes.Client,
	currentRoles *rolesState,
	cacheDirectory string,
	now time.Time,
) (*rolesState, error) {
	handleErr := func(err error) error {
		err = xerrors.Errorf("failed to update roles for slug '%s': %v", client.IdmSlug, err)
		logger.Log().Warnf("roles.Update(). %s", err)
		return err
	}

	logger.Log().Debugf("Updating roles for slug='%s'...", client.IdmSlug)

	serviceTicket, err := gate.GetServiceTicket(client.SelfTvmID)
	if err != nil {
		return nil, handleErr(err)
	}

	newRoles, err := getActualRolesForClient(
		gate.CreateRolesReader(client.IdmSlug, serviceTicket),
		currentRoles,
		now,
	)
	if newRoles.roles == nil {
		if err == nil {
			err = xerrors.New("internal error: no cache and no errors from tirole")
		}
		return nil, handleErr(err)
	}
	if err != nil {
		_ = handleErr(err)
	}

	// We should always update disk cache to prevent redundant requests for tirole-api
	filename := getCacheFileName(cacheDirectory, buildRolesFileName(client.IdmSlug))
	if err := writeCacheFile(filename, uint64(newRoles.state.lastUpdated.Unix()), newRoles.roles.GetRaw()); err != nil {
		err = xerrors.Errorf("failed to write disk cache: %s: %s", filename, err)
		logger.Log().Warnf("roles.Update(). %s", err)
		return newRoles, err
	}

	logger.Log().Debugf("Updating roles for slug='%s'... Succeed", client.IdmSlug)

	return newRoles, nil
}

func getActualRolesForClient(
	roleReader tiroleRoleReader,
	currentRoles *rolesState,
	now time.Time,
) (*rolesState, error) {
	newRoles := &rolesState{}

	var currentRevision string
	if currentRoles != nil {
		newRoles.roles = currentRoles.roles
		newRoles.state.lastUpdated = currentRoles.state.lastUpdated
		newRoles.state.lastAttempt = currentRoles.state.lastAttempt

		if !currentRoles.state.IsTimeForAttempt(now.Unix(), rolesUpdateInterval) {
			return newRoles, nil
		}
		currentRevision = currentRoles.roles.GetMeta().Revision
	}

	newRoles.state.lastAttempt = now

	roles, err := roleReader(currentRevision)
	if err == nil {
		newRoles.state.lastUpdated = now
		if roles != nil {
			newRoles.roles = roles
		}
	}

	return newRoles, err
}

func (c *rolesCache) getRolesBySlug() rolesStateBySlug {
	c.mutex.RLock()
	defer c.mutex.RUnlock()
	return c.bySlug
}

func (c *rolesCache) setRolesBySlug(r rolesStateBySlug) {
	c.mutex.Lock()
	defer c.mutex.Unlock()
	c.bySlug = r
}

func buildRolesFileName(slug string) string {
	return fmt.Sprintf(".tvm-roles.cache.%s", slug)
}

type tiroleGateImpl struct {
	tvmapiClient serviceTicketGetter
	tiroleClient rolesGetter
	tiroleTvmID  tvm.ClientID
}

func (c *tiroleGateImpl) GetServiceTicket(id tvm.ClientID) (string, error) {
	serviceTicket, er, err := c.tvmapiClient.GetTicket(id, c.tiroleTvmID)
	if er != "" {
		err = xerrors.New(er)
	}
	return string(serviceTicket), err
}

func (c *tiroleGateImpl) CreateRolesReader(slug, serviceTicket string) tiroleRoleReader {
	return func(currentRevision string) (*tvm.Roles, error) {
		return c.tiroleClient.GetRoles(slug, serviceTicket, currentRevision)
	}
}
