package tvmcache

import (
	"encoding/json"
	"errors"
	"fmt"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmcontext"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmtypes"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

const (
	ticketsUpdateInterval = 3600
)

type ServiceTicketKey struct{ Src, Dst tvm.ClientID }

type tvmTicketCache struct {
	state          cacheState
	tickets        map[ServiceTicketKey]tvmtypes.Ticket
	errors         map[ServiceTicketKey]string
	cacheDirectory string
}

func NewTicketsCache(cacheDirectory string) *tvmTicketCache {
	return &tvmTicketCache{
		cacheDirectory: cacheDirectory,
	}
}

func (c *tvmTicketCache) FetchFromDisk(config *tvmtypes.OptimizedConfig) error {
	filename := getCacheFileName(c.cacheDirectory, ticketsCacheFileName)
	tickets, tmstamp, err := tryReadTicketsCache(filename)
	if err != nil {
		return err
	}

	if time.Now().Unix()-int64(tmstamp) > ticketsUpdateInterval {
		return errors.New("tickets cache is too old, online update is required")
	}

	clients := config.GetClients()
	for num := range clients {
		for _, v := range clients[num].Dsts {
			if !checkHasSrcDst(tickets, clients[num].SelfTvmID, v.ID) {
				return errors.New("cache and configuration mismatch")
			}
		}
	}

	for v := range clients {
		c.setTicketsForClient(clients[v], tvmtypes.TicketsInfo{
			Tickets: tickets[clients[v].SelfTvmID],
			Errors:  make(map[tvm.ClientID]string)})
	}

	c.setLastUpdated(time.Unix(int64(tmstamp), 0))
	c.setLastAttempt(time.Unix(int64(tmstamp), 0))
	c.setLastError(nil)
	logger.Log().Infof("Tickets successfully read from file: %s", filename)

	return nil
}

func (c *tvmTicketCache) IsTimeForAttempt(now time.Time) bool {
	return c.state.IsTimeForAttempt(now.Unix(), ticketsUpdateInterval)
}

func (c *tvmTicketCache) GetTicketUpdateTime() time.Time {
	c.state.mtx.RLock()
	defer c.state.mtx.RUnlock()

	return c.state.lastUpdated
}

func (c *tvmTicketCache) GetTicket(src tvm.ClientID, dst tvm.ClientID) (tvmtypes.Ticket, string, error) {
	c.state.mtx.RLock()
	defer c.state.mtx.RUnlock()

	if ticket, ok := c.tickets[ServiceTicketKey{src, dst}]; ok {
		return ticket, "", nil
	}
	if er, ok := c.errors[ServiceTicketKey{src, dst}]; ok {
		return "", er, nil
	}
	return "", "", fmt.Errorf("no ticket and no error for src %d, dst %d: have you configured this pair?", src, dst)
}

func (c *tvmTicketCache) GetDiag(now time.Time) DiagState {
	return c.state.GetDiag(now.Unix(), ticketsUpdateInterval)
}

func (c *tvmTicketCache) GetTicketErrors() map[ServiceTicketKey]string {
	c.state.mtx.Lock()
	defer c.state.mtx.Unlock()

	return c.errors
}

type tvmTicketsGetter interface {
	GetTickets(secret string, src tvm.ClientID, dsts []tvmtypes.Dst) (tvmtypes.TicketsInfo, error)
}

func (c *tvmTicketCache) Update(tvmAPIClient tvmTicketsGetter, srvctx *tvmcontext.ServiceContext, clients []*tvmtypes.Client) error {
	errs := make([]error, 0)

	var wasError bool
	for v := range clients {
		if err := c.updateImpl(tvmAPIClient, srvctx, clients[v]); err != nil {
			errs = append(errs, err)
			wasError = true
		}
	}
	if !wasError && c.cacheDirectory != "" {
		c.writeToDisk()
	}

	if len(errs) != 0 {
		res := errs[0]
		for idx := 1; idx < len(errs); idx++ {
			res = xerrors.Errorf("%v. %v", res, errs[idx])
		}
		return res
	}

	return nil
}

func (c *tvmTicketCache) updateImpl(tvmAPIClient tvmTicketsGetter, srvctx *tvmcontext.ServiceContext, client *tvmtypes.Client) error {
	if len(client.Dsts) == 0 {
		c.setLastAttempt(time.Now())
		c.setLastUpdated(time.Now())
		return nil
	}

	logger.Log().Infof("Updating tickets for src=%d...", client.SelfTvmID)
	var dsts []tvmtypes.Dst

	for _, v := range client.Dsts {
		dsts = append(dsts, v)
	}

	ticks, err := tvmAPIClient.GetTickets(
		client.Secret,
		client.SelfTvmID,
		dsts,
	)

	c.setLastAttempt(time.Now())
	if err != nil {
		logger.Log().Warnf("Tickets update error %s", err)
		c.setLastError(err)
		return err
	}
	if len(ticks.Tickets) == 0 {
		err = fmt.Errorf("failed to get any ticket for src=%d", client.SelfTvmID)
		c.setLastError(err)
		return err
	}

	if err = c.checkServiceTickets(srvctx, ticks); err != nil {
		c.setLastError(err)
		return err
	}

	c.setTicketsForClient(client, ticks)

	c.setLastUpdated(time.Now())
	c.setLastError(nil)
	logger.Log().Infof("Updating tickets for src=%d... Succeed", client.SelfTvmID)

	return nil
}

func (c *tvmTicketCache) checkServiceTickets(srvctx *tvmcontext.ServiceContext, ticks tvmtypes.TicketsInfo) error {
	for _, ticket := range ticks.Tickets {
		if _, err := srvctx.CheckTicketWithoutDst(string(ticket)); err != nil {
			return fmt.Errorf("failed to check ticket from tvm-api: %s: %s", err.Error(), ticket)
		}
	}

	return nil
}

func (c *tvmTicketCache) setLastUpdated(tm time.Time) {
	c.state.mtx.Lock()
	defer c.state.mtx.Unlock()

	c.state.lastUpdated = tm
}

func (c *tvmTicketCache) setLastAttempt(tm time.Time) {
	c.state.mtx.Lock()
	defer c.state.mtx.Unlock()

	c.state.lastAttempt = tm
}

func (c *tvmTicketCache) setLastError(err error) {
	c.state.mtx.Lock()
	defer c.state.mtx.Unlock()

	c.state.lastError = err
}

func (c *tvmTicketCache) setTicketsForClient(client *tvmtypes.Client, ticks tvmtypes.TicketsInfo) {
	c.state.mtx.Lock()
	defer c.state.mtx.Unlock()

	if c.tickets == nil {
		c.tickets = make(map[ServiceTicketKey]tvmtypes.Ticket)
	}
	c.errors = make(map[ServiceTicketKey]string)

	for dstid, ticket := range ticks.Tickets {
		c.tickets[ServiceTicketKey{client.SelfTvmID, dstid}] = ticket
	}
	for dstid, err := range ticks.Errors {
		c.errors[ServiceTicketKey{client.SelfTvmID, dstid}] = err
	}
}

func (c *tvmTicketCache) writeToDisk() {
	tickets := c.copyTickets()
	filename := getCacheFileName(c.cacheDirectory, ticketsCacheFileName)

	_ = saveTicketsCache(filename, tickets, uint64(time.Now().Unix()))
}

func (c *tvmTicketCache) copyTickets() map[ServiceTicketKey]tvmtypes.Ticket {
	c.state.mtx.RLock()
	defer c.state.mtx.RUnlock()

	result := make(map[ServiceTicketKey]tvmtypes.Ticket)
	for k, v := range c.tickets {
		result[k] = v
	}

	return result
}

type ticketsCacheFileStruct map[tvm.ClientID]tvmtypes.TicketsForOneClient

func saveTicketsCache(filename string, tickets map[ServiceTicketKey]tvmtypes.Ticket, tmstamp uint64) error {
	t := make(map[tvm.ClientID]tvmtypes.TicketsForOneClient)

	for k, v := range tickets {
		if _, ok := t[k.Src]; !ok {
			t[k.Src] = make(tvmtypes.TicketsForOneClient)
		}
		t[k.Src][k.Dst] = v
	}

	data, err := json.Marshal(t)
	if err != nil {
		logger.Log().Debugf("Failed to serialize json: %s", err)
		return err
	}

	return writeCacheFile(filename, tmstamp, data)
}

func tryReadTicketsCache(filename string) (ticketsCacheFileStruct, uint64, error) {
	data, tmstamp, err := readCacheFile(filename)
	if err != nil {
		return nil, 0, err
	}

	var t ticketsCacheFileStruct
	err = json.Unmarshal(data, &t)
	if err != nil {
		return nil, 0, errors.New("failed to parse json in cache")
	}

	return t, tmstamp, nil
}
