package tvmcontext

import (
	"encoding/base64"
	"strings"

	"github.com/golang/protobuf/proto"
	"golang.org/x/xerrors"

	"a.yandex-team.ru/library/cpp/tvmauth/src/protos"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/crypto"
)

const (
	KeysFormatVersion = "1"
	KeysDelimiter     = ":"
)

type keyID uint32

func (k keyID) IsTesting() bool {
	return k < 20
}

func checkKeyEnvironmentMismatch(ticketID keyID, ctxID keyID) error {
	if ticketID.IsTesting() && !ctxID.IsTesting() {
		return xerrors.Errorf(
			"You are trying to use tickets from 'unittest'-mode with keys from TVM-API production. Key id: %d",
			ticketID,
		)
	}

	if !ticketID.IsTesting() && ctxID.IsTesting() {
		return xerrors.Errorf(
			"You are trying to use tickets from TVM-API production with keys from 'unittest'-mode. Key id: %d",
			ticketID,
		)
	}

	return nil
}

func ParseKeys(keys string) (*protos.Keys, error) {
	t := strings.TrimSpace(keys)
	tokens := strings.Split(t, KeysDelimiter)
	if len(tokens) != 2 {
		return nil, ErrorInvalidKeysFormat
	}

	if tokens[0] != KeysFormatVersion {
		return nil, errorUnsupportedKeysVersion
	}

	data, err := base64.RawURLEncoding.DecodeString(tokens[1])
	if err != nil {
		return nil, errorInvalidKeysBase64
	}

	protokeys := &protos.Keys{}
	if err := proto.Unmarshal(data, protokeys); err != nil {
		return nil, ErrorInvalidKeysProtobuf
	}

	return protokeys, nil
}

func parseKey(k *protos.General) (*crypto.RWPublicKey, error) {
	t := k.GetType()
	if t != protos.KeyType_RabinWilliams {
		return nil, xerrors.Errorf("unsupported public key type: %s", t)
	}

	rw, err := crypto.RWPublicKeyFromBin(k.GetBody())
	if err != nil {
		return nil, xerrors.Errorf("failed to parse public key: %w", err)
	}

	return rw, nil
}
