package tvmcontext

import (
	"encoding/base64"
	"fmt"
	"strings"
	"time"

	"a.yandex-team.ru/library/cpp/tvmauth/src/protos"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/crypto"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/errs"
)

type publicTvmKeys map[keyID]*crypto.RWPublicKey

type ServiceContext struct {
	keys         publicTvmKeys
	keyIDExample keyID
}

type CheckedServiceTicket struct {
	SrcID     tvm.ClientID
	DstID     tvm.ClientID
	IssuerUID *uint64
	DbgInfo   string
	LogInfo   string
}

func NewServiceContext(keys string) (*ServiceContext, error) {
	parsedKeys, err := ParseKeys(keys)
	if err != nil {
		return nil, err
	}

	var id keyID
	rwKeys := make(publicTvmKeys)
	for _, k := range parsedKeys.GetTvm() {
		rw, err := parseKey(k.GetGen())
		if err != nil {
			return nil, err
		}

		id = keyID(k.GetGen().GetId())
		rwKeys[id] = rw
	}

	if len(rwKeys) == 0 {
		return nil, errorNoPublicKeysService
	}

	return &ServiceContext{keys: rwKeys, keyIDExample: id}, nil
}

func (s *ServiceContext) checkTicketSignature(ticket *parsedTicket) error {
	keyid := keyID(ticket.Ticket.GetKeyId())
	key, ok := s.keys[keyid]
	if !ok {
		if err := checkKeyEnvironmentMismatch(keyid, s.keyIDExample); err != nil {
			return err
		}
		return fmt.Errorf("key id for service ticket not found: %d. Maybe keys are too old", ticket.Ticket.GetKeyId())
	}

	signature, err := base64.RawURLEncoding.DecodeString(ticket.Signature)
	if err != nil {
		return errorInvalidSignatureBase64
	}

	result, err := key.VerifyMsg(ticket.Msg, signature)
	if err != nil {
		logThrottled(err, ticket.Msg+ticket.Signature)
		return err
	}

	if !result {
		logThrottled(errorIncorrectSignature, ticket.Msg+ticket.Signature)
		return errorIncorrectSignature
	}

	return nil
}

func (s *ServiceContext) CheckTicket(str string, expectedDst tvm.ClientID) (*CheckedServiceTicket, error) {
	checked, err := s.CheckTicketWithoutDst(str)
	if err != nil {
		return nil, err
	}

	if checked.DstID != expectedDst {
		return nil, &errs.Forbidden{
			Message:       fmt.Sprintf("Wrong ticket dst, expected %d, got %d", expectedDst, checked.DstID),
			DebugString:   checked.DbgInfo,
			LoggingString: checked.LogInfo,
		}
	}

	return checked, nil
}

func (s *ServiceContext) CheckTicketWithoutDst(str string) (*CheckedServiceTicket, error) {
	return s.checkTicketWithoutDstImpl(str, time.Now())
}

func (s *ServiceContext) checkTicketWithoutDstImpl(str string, now time.Time) (*CheckedServiceTicket, error) {
	parsedtkt, err := parseFromStr(str)
	if err != nil {
		return nil, &errs.Forbidden{Message: err.Error(), LoggingString: str}
	}

	if parsedtkt.Type != TicketTypeServiceTicket {
		return nil, &errs.Forbidden{
			Message:       "wrong ticket type, service-ticket is expected",
			LoggingString: parsedtkt.Msg,
		}
	}

	if parsedtkt.Ticket.GetExpirationTime() < now.Unix() {
		return nil, &errs.Forbidden{
			Message:       fmt.Sprintf("expired ticket, exp_time %d, now %d", (*parsedtkt).Ticket.GetExpirationTime(), now.Unix()),
			DebugString:   makeDebugStringForServiceTicket(parsedtkt.Ticket),
			LoggingString: parsedtkt.Msg,
		}
	}

	if err := s.checkTicketSignature(parsedtkt); err != nil {
		return nil, &errs.Forbidden{
			Message:       err.Error(),
			DebugString:   makeDebugStringForServiceTicket(parsedtkt.Ticket),
			LoggingString: parsedtkt.Msg,
		}
	}

	return &CheckedServiceTicket{
		SrcID:     tvm.ClientID(parsedtkt.Ticket.Service.GetSrcClientId()),
		DstID:     tvm.ClientID(parsedtkt.Ticket.Service.GetDstClientId()),
		IssuerUID: parsedtkt.Ticket.Service.IssuerUid,
		DbgInfo:   makeDebugStringForServiceTicket(parsedtkt.Ticket),
		LogInfo:   parsedtkt.Msg,
	}, nil
}

func makeDebugStringForServiceTicket(ticket *protos.Ticket) string {
	if ticket == nil {
		return ""
	}

	return fmt.Sprintf(
		"ticket_type=service;expiration_time=%d;src=%d;dst=%d;scope=%s;",
		ticket.GetExpirationTime(),
		ticket.Service.GetSrcClientId(),
		ticket.Service.GetDstClientId(),
		strings.Join(ticket.Service.GetScopes(), ","),
	)
}
