package tvmcontext

import (
	"encoding/base64"
	"errors"
	"strings"

	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/library/cpp/tvmauth/src/protos"
)

const (
	TvmTicketsVersion3 = "3"
	TvmTicketDelimiter = ":"
)

type TicketType int

const (
	TicketTypeUserTicket TicketType = iota
	TicketTypeServiceTicket
)

const (
	ServiceTicketSrt = "serv"
	UserTicketStr    = "user"
)

var errorMalformed = errors.New("invalid ticket format")
var errorUnsupportedVersion = errors.New("unsupported ticket version")
var errorUnsupportedType = errors.New("unsupported ticket type")
var errorInvalidBase64 = errors.New("invalid base64 in ticket")
var errorInvalidProtobuf = errors.New("invalid protobuf in ticket")
var errorWrongTicketTypeUsr = errors.New("wrong ticket type (usr)")
var errorWrongTicketTypeSrv = errors.New("wrong ticket type (srv)")
var errorMissingKeyid = errors.New("wrong ticket format (keyid)")
var errorMissingExpTime = errors.New("wrong ticket format (expiration time)")

type parsedTicket struct {
	Version   string
	Proto     string
	Signature string
	Type      TicketType
	Msg       string
	Ticket    *protos.Ticket
}

func parseFromStr(str string) (*parsedTicket, error) {
	tokens := strings.Split(strings.TrimSpace(str), TvmTicketDelimiter)
	if len(tokens) != 4 {
		return nil, errorMalformed
	}

	if tokens[0] != TvmTicketsVersion3 {
		return nil, errorUnsupportedVersion
	}

	result := parsedTicket{Version: tokens[0]}

	switch tokens[1] {
	case ServiceTicketSrt:
		result.Type = TicketTypeServiceTicket
	case UserTicketStr:
		result.Type = TicketTypeUserTicket
	default:
		return nil, errorUnsupportedType
	}

	result.Proto = tokens[2]
	result.Signature = tokens[3]
	result.Msg = strings.Join(tokens[0:3], TvmTicketDelimiter) + TvmTicketDelimiter

	data, err := base64.RawURLEncoding.DecodeString(result.Proto)
	if err != nil {
		return nil, errorInvalidBase64
	}

	ticket := &protos.Ticket{}
	if err := proto.Unmarshal(data, ticket); err != nil {
		return nil, errorInvalidProtobuf
	}

	if result.Type == TicketTypeServiceTicket && ticket.GetService() == nil {
		return nil, errorWrongTicketTypeSrv
	}

	if result.Type == TicketTypeUserTicket && ticket.GetUser() == nil {
		return nil, errorWrongTicketTypeUsr
	}

	if ticket.GetKeyId() == 0 {
		return nil, errorMissingKeyid
	}

	if ticket.GetExpirationTime() == 0 {
		return nil, errorMissingExpTime
	}

	result.Ticket = ticket
	return &result, nil
}
