package tvmsrv

import (
	"fmt"
	"net/url"
	"os"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmtypes"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type ClientView struct {
	Secret    string                  `json:"secret,omitempty"`
	SelfTvmID tvm.ClientID            `json:"self_tvm_id"`
	Dsts      map[string]tvmtypes.Dst `json:"dsts,omitempty"`
	IdmSlug   string                  `json:"roles_for_idm_slug,omitempty"`
}

type SolomonView struct {
	BindHost     string       `json:"bind_host,omitempty"`
	Port         uint16       `json:"port"`
	StageTVMID   tvm.ClientID `json:"stage_tvm_id"`
	SolomonTVMID tvm.ClientID `json:"solomon_tvm_id"`
}

type BackendsView struct {
	TvmURL      *string       `json:"tvm_url,omitempty"`
	TiroleURL   *string       `json:"tirole_url,omitempty"`
	TiroleTvmID *tvm.ClientID `json:"tirole_tvm_id,omitempty"`
}

type ConfigView struct {
	BbEnvType      *tvm.BlackboxEnv      `json:"BbEnvType,omitempty"`
	Clients        map[string]ClientView `json:"clients"`
	Port           *uint16               `json:"port,omitempty"`
	PingPort       *uint16               `json:"ping_port,omitempty"`
	Solomon        *SolomonView          `json:"solomon,omitempty"`
	Backends       *BackendsView         `json:"backends"`
	UseSystemCerts *bool                 `json:"use_system_certs,omitempty"`
}

func (c *ConfigView) toConfig() (*tvmtypes.Config, error) {
	var err error

	env := tvm.BlackboxEnv(0)
	if c.BbEnvType != nil {
		env = *c.BbEnvType
		if !tvmtypes.IsValidEnv(env) {
			logger.Log().Errorf("wrong blackbox env type: %d", env)
			return nil, xerrors.Errorf("wrong blackbox env type: %d", env)
		}
	}

	backends := c.Backends
	if backends == nil {
		backends = &BackendsView{}
	}

	tvmURL := &url.URL{Scheme: "https", Host: "tvm-api.yandex.net"}
	if backends.TvmURL != nil {
		tvmURL, err = url.Parse(*backends.TvmURL)
		if err != nil {
			return nil, err
		}
	}
	if testURL := os.Getenv("__TEST_TVM_API_URL"); testURL != "" {
		tvmURL, err = url.Parse(testURL)
		if err != nil {
			return nil, err
		}
	}

	tiroleURL := &url.URL{Scheme: "https", Host: "tirole-api.yandex.net"}
	if backends.TiroleURL != nil {
		tiroleURL, err = url.Parse(*backends.TiroleURL)
		if err != nil {
			return nil, err
		}
	}
	tiroleTvmID := tvm.ClientID(2028120)
	if backends.TiroleTvmID != nil {
		tiroleTvmID = *backends.TiroleTvmID
	}
	if testURL := os.Getenv("__TEST_TIROLE_URL"); testURL != "" {
		tiroleURL, err = url.Parse(testURL)
		if err != nil {
			return nil, err
		}
		tiroleTvmID = tvm.ClientID(1000001)
	}

	cfg := &tvmtypes.Config{
		BbEnvType: env,
		Backends: tvmtypes.BackendsConfig{
			TvmURL:      tvmURL,
			TiroleURL:   tiroleURL,
			TiroleTvmID: tiroleTvmID,
		},
		Clients: map[string]tvmtypes.Client{},
	}

	for k, v := range c.Clients {
		if v.Dsts == nil {
			v.Dsts = make(map[string]tvmtypes.Dst)
		}
		if v.IdmSlug != "" {
			v.Dsts[fmt.Sprintf("%d", tiroleTvmID)] = tvmtypes.Dst{ID: tiroleTvmID}
		}

		cfg.Clients[k] = tvmtypes.Client{
			Alias:     k,
			Secret:    v.Secret,
			SelfTvmID: v.SelfTvmID,
			Dsts:      v.Dsts,
			IdmSlug:   v.IdmSlug,
		}
	}

	solomon := c.Solomon
	if solomon != nil {
		cfg.Solomon = &tvmtypes.SolomonConfig{
			BindHost:     solomon.BindHost,
			Port:         solomon.Port,
			StageTVMID:   solomon.StageTVMID,
			SolomonTVMID: solomon.SolomonTVMID,
		}
	}

	cfg.UseSystemCerts = true
	if c.UseSystemCerts != nil {
		cfg.UseSystemCerts = *c.UseSystemCerts
	}

	return cfg, nil
}
