package tvmsrv

import (
	"net/http/pprof"

	"github.com/labstack/echo/v4"

	tvm "a.yandex-team.ru/passport/infra/daemons/tvmtool/internal"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/middleware"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmversion"
)

type Handlers struct {
	Main    *echo.Echo
	Ping    *echo.Echo
	Solomon *echo.Echo
}

func createHandlers(tvmstruct *tvm.Tvm, auth string) *Handlers {
	res := &Handlers{
		Main:    echo.New(),
		Ping:    echo.New(),
		Solomon: echo.New(),
	}

	handlePing := func(e *echo.Echo) {
		e.GET(
			"/tvm/ping",
			tvmstruct.GetPingHandler(),
			middleware.NewErrorHandleMiddlewareV1(),
		)
		e.GET(
			"/ping",
			tvmstruct.GetPingHandlerV2(),
			middleware.NewErrorHandleMiddlewareV2(),
		)
	}
	handlePing(res.Main)
	handlePing(res.Ping)

	for _, pair := range []struct {
		path   string
		handle echo.HandlerFunc
	}{
		{"/tvm/tickets", tvmstruct.GetTicketHandler()},
		{"/tvm/keys", tvmstruct.GetKeysHandler()},
		{"/tvm/checksrv", tvmstruct.GetCheckSrvTicketHandler()},
		{"/tvm/checkusr", tvmstruct.GetCheckUsrHandler()},
		{"/tvm/private_api/__meta__", tvmstruct.GetMetaHandler()},
		{"/tvm/cache/force_update", tvmstruct.GetForceUpdateHandler()},
		{"/debug/pprof/", echo.HandlerFunc(func(ctx echo.Context) error {
			pprof.Index(ctx.Response(), ctx.Request())
			return nil
		})},
	} {
		res.Main.GET(
			pair.path,
			pair.handle,
			middleware.NewErrorHandleMiddlewareV1(),
			middleware.NewAuthMiddleware(auth),
		)
	}

	for _, pair := range []struct {
		path   string
		handle echo.HandlerFunc
	}{
		{"/v2/check", tvmstruct.GetCheckV2()},
		{"/v2/keys", tvmstruct.GetKeysHandler()},
		{"/v2/roles", tvmstruct.GetRolesV2()},
		{"/v2/tickets", tvmstruct.GetTicketHandlerV2()},
	} {
		res.Main.GET(
			pair.path,
			pair.handle,
			middleware.NewErrorHandleMiddlewareV2(),
			middleware.NewAuthMiddleware(auth),
		)
	}

	res.Solomon.GET(
		"/tvm/solomon",
		tvmstruct.GetSolomonMetricsHandler(),
		middleware.NewErrorHandleMiddlewareV1(),
		tvmstruct.GetSolomonAuthMiddleware(),
	)

	routes := res.Main.Routes()
	routes = append(routes, res.Ping.Routes()...)
	routes = append(routes, res.Solomon.Routes()...)

	res.addMiddlewares(
		middleware.NewServerVersionMiddleware(tvmversion.GetServerVersion()),
		middleware.NewSolomonHandlerMetricsMiddleware(routes, tvmstruct.GetSolomonReg()),
		middleware.NewNoSniffMiddleware(),
	)

	return res
}

func (h *Handlers) addMiddlewares(version, rpsByPath, nosniff echo.MiddlewareFunc) {
	add := func(e *echo.Echo) {
		e.Pre(version)
		e.Use(rpsByPath, nosniff)
	}

	add(h.Main)
	add(h.Ping)
	add(h.Solomon)
}
