package tvmsrv

import (
	"crypto/tls"
	"crypto/x509"
	"fmt"
	"net"
	"net/http"

	"golang.org/x/net/context"

	"a.yandex-team.ru/library/go/certifi"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/httpclientmock"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type ListenedInterfaces struct {
	Main    []Listener
	Ping    []Listener
	Solomon []Listener
}

type Listener struct {
	listener  net.Listener
	printable string
}

func listenToolInterfaces(interfaceToBind string, config *ConfigView, port uint16) (*ListenedInterfaces, error) {
	res := &ListenedInterfaces{}
	var err error

	res.Main, err = startInterfaces(interfaceToBind, port)
	if err != nil {
		return nil, err
	}

	if config.PingPort != nil {
		ping, err := listenInterface(interfaceToBind, *config.PingPort)
		if err != nil {
			return nil, fmt.Errorf("failed to use port (ping interface): %d: %s", *config.PingPort, err)
		}
		res.Ping = []Listener{ping}
	}

	if config.Solomon != nil {
		solomon, err := listenInterface(config.Solomon.BindHost, config.Solomon.Port)
		if err != nil {
			return nil, fmt.Errorf("failed to use port (solomon interface): %d: %s", config.Solomon.Port, err)
		}
		res.Solomon = []Listener{solomon}
	}

	return res, nil
}

func startInterfaces(interfaceToBind string, port uint16) ([]Listener, error) {
	toBind := makeInterfacesArray(interfaceToBind)

	var interfaces []Listener
	for idx := range toBind {
		in, err := listenInterface(toBind[idx], port)
		if err != nil {
			return nil, fmt.Errorf("failed to use port (main interface) %d: %s", port, err)
		}
		interfaces = append(interfaces, in)
	}

	return interfaces, nil
}

func listenInterface(interfaceToBind string, port uint16) (Listener, error) {
	lc := getListenConfig()

	printable := fmt.Sprintf("[%s]:%d", interfaceToBind, port)
	listener, err := lc.Listen(context.Background(), "tcp", printable)

	return Listener{listener, printable}, err
}

func createHTTPTransport(useSystemCerts bool) *http.Transport {
	var certPool *x509.CertPool
	var err error

	if useSystemCerts {
		certPool, err = certifi.NewCertPoolSystem()
	} else {
		logger.Log().Debug("Skip system certs, trying to use only built-in certs")
		certPool, err = certifi.NewCertPoolBundled()
	}
	if err != nil {
		logger.Log().Warnf("failed to create cert pool: %v", err)
		return nil
	}

	return &http.Transport{
		TLSClientConfig: &tls.Config{
			RootCAs: certPool,
		},
	}
}

func createHTTPClient(verbose bool, trans *http.Transport) *http.Client {
	if verbose {
		return httpclientmock.NewDumpClientWithLogger(
			logger.Log(),
			trans,
		)
	}

	return &http.Client{
		Transport: trans,
	}
}
