package tvmsrv

import (
	"os"
	"os/signal"
	"syscall"

	"golang.org/x/net/context"

	tvm "a.yandex-team.ru/passport/infra/daemons/tvmtool/internal"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

func StartService(
	config *ConfigView,
	port uint16,
	auth string,
	verbose bool,
	cacheDir string,
	interfaceToBind string,
	unittest bool,
	unittestRolesDir string,
) error {
	if len(config.Clients) == 0 { // TODO: PASSP-23049
		logger.Log().Errorf("No one client was found in config. Later it will be fatal error: PASSP-23049")
		// return errors.New("no one client was found in config")
	}

	signalChannel := make(chan os.Signal, 3)
	signal.Notify(signalChannel, os.Interrupt, syscall.SIGTERM)

	// Listen ports before other actions: they can be busy
	listenedInterfaces, err := listenToolInterfaces(interfaceToBind, config, port)
	if err != nil {
		return err
	}

	cfg, err := config.toConfig()
	if err != nil {
		return err
	}
	if err := cfg.CheckTiroleFeature(cacheDir != ""); err != nil {
		return err
	}

	tvmstruct := tvm.NewTvm(
		createHTTPClient(verbose, createHTTPTransport(cfg.UseSystemCerts)),
		cfg,
		verbose,
		cacheDir,
	)

	if err := tvmstruct.Start(unittest, unittestRolesDir); err != nil {
		return err
	}

	servers, errChannel := startServers(
		listenedInterfaces,
		createHandlers(tvmstruct, auth),
	)

	var serverError error
	select {
	case sig := <-signalChannel:
		switch sig {
		case syscall.SIGABRT:
			logger.Log().Errorf("Stopping daemon: http interface stopped")
		case syscall.SIGTERM:
			logger.Log().Infof("Stopping daemon: SIGTERM")
		case os.Interrupt:
			logger.Log().Infof("Stopping daemon: SIGINT")
		}

	case serverError = <-errChannel:
		logger.Log().Errorf("Stopping daemon: http interface stopped")
	}

	logger.Log().Infof("Shutting down the service...")

	for _, serv := range servers {
		if err := serv.HTTP.Shutdown(context.Background()); err != nil {
			logger.Log().Warnf("Shutting down %s interface: %s", serv.Name, err)
		}
	}

	tvmstruct.Stop()

	return serverError
}
