#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/tvmauth/unittest.h>
#include <library/cpp/tvmauth/client/facade.h>

#include <util/stream/file.h>
#include <util/string/builder.h>
#include <util/string/subst.h>
#include <util/system/fs.h>
#include <util/system/shellcommand.h>

using namespace NTvmAuth;

static const TString SRV_TICKET_INV =
    "3:serv:CBAQ__________9_IgYIexCUkQY:GioCM49Ob6_f80y6FY0XBVN4hLXuMlFeyMvIMiDuQnZkbkLpRp"
    "QOuQo5YjWoBjM0Vf-XqOm8B7xtrvxSYHDD7Q4OatN2l-Iwg7i71lE3scUeD36x47st3nd0OThvtjrFx_D8mw_"
    "c0GT5KcniZlqq1SjhLyAk1b_zJsx8viRAhCU";

static const TString SRV_TICKET =
    "3:serv:CBAQ__________9_IgQIexAq:CeYHbo4MJQajoJluQVkCs8KPZGh454Xqdk8wBylfa_2xmQ2euarOV"
    "OGIg4q9OULydSXghWWcbhCMfJiNkp3ALeFVA0HctTjowNqbi5Kg8LesNQbbLeEn4DBX2psrxn9Ifu_ZHFAErj"
    "548jWB6ajicWsNNLXSF7RNH1I6kg98_WM";

class Logger: public ILogger {
public:
    void Log(int lvl, const TString& msg) override {
        Stream << lvl << ": " << msg << Endl;
    }

    TStringStream Stream;
};

Y_UNIT_TEST_SUITE(TvmToolClient) {
    static TIntrusivePtr<Logger> CreateLogger() {
        return new Logger;
    }

    static NTvmTool::TClientSettings CreateSettings(const TString& alias = "me") {
        TFileInput port("tvmtool.port");
        TFileInput authtoken("tvmtool.authtoken");

        NTvmTool::TClientSettings s(alias);
        s.SetPort(IntFromString<int, 10>(port.ReadAll()));
        s.SetAuthToken(authtoken.ReadAll());
        return s;
    }

    Y_UNIT_TEST(rowCtxBadPort) {
        auto logger = CreateLogger();
        NTvmTool::TClientSettings s("me");
        s.SetPort(0);
        s.SetAuthToken("qwerty");
        UNIT_ASSERT_EXCEPTION_CONTAINS_C(TTvmClient(s, logger),
                                         TNonRetriableException,
                                         "can not connect to ",
                                         logger->Stream.Str());
    }

    Y_UNIT_TEST(BadSrc) {
        auto logger = CreateLogger();
        NTvmTool::TClientSettings s = CreateSettings("qwerty");
        UNIT_ASSERT_EXCEPTION_CONTAINS_C(TTvmClient(s, logger),
                                         TNonRetriableException,
                                         R"(Alias 'qwerty' not found in meta info)",
                                         logger->Stream.Str());
    }

    Y_UNIT_TEST(OkOperation) {
        auto logger = CreateLogger();
        NTvmTool::TClientSettings s = CreateSettings();
        TTvmClient c(s, logger);
        UNIT_ASSERT_EQUAL(TClientStatus::Ok, c.GetStatus().GetCode());

        UNIT_ASSERT(!c.CheckServiceTicket(SRV_TICKET_INV));
        UNIT_ASSERT(c.CheckServiceTicket(SRV_TICKET));

        UNIT_ASSERT_EXCEPTION_CONTAINS(c.GetServiceTicketFor("ololo"),
                                       TBrokenTvmClientSettings,
                                       "Destination 'ololo' was not specified in settings. Check your settings (if you use Qloud/YP/tvmtool - check it's settings)");
        UNIT_ASSERT_EXCEPTION_CONTAINS(c.GetServiceTicketFor(123),
                                       TBrokenTvmClientSettings,
                                       "Destination '123' was not specified in settings. Check your settings (if you use Qloud/YP/tvmtool - check it's settings)");

        TString he = c.GetServiceTicketFor("he");
        UNIT_ASSERT(TServiceContext::CheckingFactory(100500, NUnittest::TVMKNIFE_PUBLIC_KEYS).Check(he));
        TString she = c.GetServiceTicketFor(100501);
        UNIT_ASSERT(TServiceContext::CheckingFactory(100501, NUnittest::TVMKNIFE_PUBLIC_KEYS).Check(she));
    }
}
