#include <passport/infra/libs/cpp/tvm/signer/signer.h>

#include <library/cpp/getopt/small/last_getopt.h>
#include <library/cpp/http/simple/http_client.h>
#include <library/cpp/openssl/init/init.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/tvmauth/unittest.h>
#include <library/cpp/tvmauth/deprecated/service_context.h>
#include <library/cpp/tvmauth/src/utils.h>

#include <util/stream/file.h>

#include <thread>
#include <vector>

Y_UNIT_TEST_SUITE(Fuzzing) {
    ui16 GetPort() {
        TFileInput port("tvmtool.port");
        return IntFromString<int, 10>(port.ReadAll());
    }

    TString GetAuthToken() {
        TFileInput authtoken("tvmtool.authtoken");
        return authtoken.ReadAll();
    }

    TStringStream Run(int tries,
                      const NTvmAuth::NRw::TRwPrivateKey& pk,
                      const NTvmAuth::TServiceContext& ctx,
                      const NTvmAuth::TTvmId dst) {
        TStringStream err;
        const TString authtoken = GetAuthToken();

        TKeepAliveHttpClient cl("localhost", GetPort());
        if (auto code = cl.DoGet("/tvm/ping"); 200 != code) {
            err << "Ping: " << code << Endl;
            return err;
        }

        NPassport::NTicketSigner::TServiceSigner signer;
        signer.SetSides(100500, dst);

        while (tries-- > 0) {
            TStringStream out;
            TKeepAliveHttpClient::THttpCode code;
            TString ticket = signer.SerializeV3(pk, std::numeric_limits<time_t>::max());

            const NTvmAuth::ETicketStatus status = ctx.Check(ticket).GetStatus();
            if (status != NTvmAuth::ETicketStatus::Ok) {
                err << "------------------ Self check failed: " << NTvmAuth::StatusToString(status) << Endl;
                err << "Ticket: " << ticket << Endl;
            }

            try {
                code = cl.DoGet("/tvm/checksrv",
                                &out,
                                {
                                    {"Authorization", authtoken},
                                    {"X-Ya-Service-Ticket", ticket},
                                });
            } catch (const std::exception& e) {
                Cerr << "Request failed: " << e.what() << Endl;
                continue;
            }

            if (code != HTTP_OK) {
                err << "----------------- Code: " << code << Endl;
                err << "Ticket: " << ticket << Endl;
                err << "Response: " << out.Str() << Endl;
            }
        }

        return err;
    }

    Y_UNIT_TEST(crossCheck) {
        InitOpenSSL();

        const TString privKey = NTvmAuth::NUtils::Base64url2bin("MIICVQKBgQC4a4oW39xKYw0EtrAkB2s6BYDdJwWxPXnrJ5xU380BC129oJyVPXp5lf2g6mhyo4LSyD6QoJ-NJR65-ZOprx-TnXnciOnAsxPjfhhad5wKtDQycjHtH4bSEQ__wm09LpXL5HqMOubm11n8OnQQtoEf9-K3THvy_0xkSzFow1C2zQJBANJufhSF9qTZnFlA73M3Mofzhf2AqVqe2L2uN5S9mQyhx6o8zzBm7pGiMiIea9dmOhLyS_BihHkmq4dIw40a25MCQQDgWxGhgsLCiR8R7qd7ixC669xRXyimJz5QYpWKeXhZH4fUjLIbgRRQVhKezMunkAs81yPSVwFyzla8kFTYA2AfAkEAigT7h01LzA_rL9xSyd_I17dSZkVisp7vdyxnrjD1iZqX2IPF9RTrEi7sboCaaP1hS2-G1vs1he3QNdcoTyEGegJAHAtiNDBYWFEj4j3U73FiF117iivlFMTnygxSsU8vCyPw-pGWQ3AiigrCU9mZdPIBZ5rkekrgLlnK15IKmwBsBAJAGk3PwpC-1Jsziygd7mbmUP5wv7AVK1PbF7XG8pezIZQ49UeZ5gzd0jRGREPNeuzHQl5JfgxQjyTVcOkYcaNbcgJATBRJcYoDdz5Y4Wq3Aui8cu8eMFe7iL1Q63twjW47OBjkmLKYYckGryfPHr9IISSazZHAnphq3TkCDZWRYhoMDQJASwsbzFCz7PcNqmQzPBi3foWh7PsYGgpwYINuBS4VnWhIlSaUz3y4HYIyBywVYkIfHfatfRtoT3pNrcosXH0BYg");
        int id = 16;
        const NTvmAuth::NRw::TRwPrivateKey pk(privKey, id);

        const NTvmAuth::TTvmId dst = 42;
        const NTvmAuth::TServiceContext ctx =
            NTvmAuth::TServiceContext::CheckingFactory(dst, NTvmAuth::NUnittest::TVMKNIFE_PUBLIC_KEYS);

        std::mutex m;
        TStringStream err;

        std::vector<std::thread> thr;
        thr.reserve(100);
        for (int i = 0; i < 100; ++i) {
            thr.push_back(std::thread([&m, &err, &pk, &ctx]() {
                TStringStream lerr;

                try {
                    lerr = Run(2000, pk, ctx, dst);
                } catch (const std::exception& e) {
                    Cerr << "Thread exiting: " << e.what() << Endl;
                }

                std::unique_lock lock(m);
                err << lerr.Str();
            }));
        }

        for (std::thread& t : thr) {
            t.join();
        }

        Cerr << err.Str();
        UNIT_ASSERT_STRINGS_EQUAL("", err.Str());
    }
}
