#include <passport/infra/daemons/xunistater/src/config/list.h>
#include <passport/infra/daemons/xunistater/src/config/parsers.h>
#include <passport/infra/daemons/xunistater/src/config/validator.h>
#include <passport/infra/daemons/xunistater/src/parsers/tskv.h>
#include <passport/infra/daemons/xunistater/src/parsers/tsv.h>

#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/xml/config.h>

#include <library/cpp/getopt/small/last_getopt.h>
#include <library/cpp/getopt/small/modchooser.h>
#include <library/cpp/xml/encode/encodexml.h>

using namespace NPassport;
using namespace NPassport::NXunistater;

static const TString CONFIG_XPATH = R"(/config/component)";

class TModeTskvCondition: public TMainClass {
    static void Run(const TString& condition) {
        Cout << "Using condition: " << condition << Endl << Endl;
        TString exp = NTskv::NConds::TrimExpression(condition);
        NTskv::NConds::ValidateExpression(exp);
        NTskv::NConds::TCondition cond = NTskv::NConds::ExpressionToCondition(exp);
        Cout << "Serialized state-machine: " << cond->Serialize() << Endl << Endl;

        class TParser: public NTskv::TTskvParser {
        public:
            using TTskvParser::ParseLine;
        };

        TString tmp;
        while (Cin.ReadLine(tmp) > 0) {
            Cout << "Line: '" << tmp << "'" << Endl;

            NTskv::TKeyValue map;
            TString buf;
            TErrorMsg err = TParser::ParseLine(tmp, map, buf);

            if (err) {
                Cout << "Parsing failed: " << err << Endl << Endl;
                continue;
            }

            bool res = cond->Match(map);
            Cout << "Parsing succeed. Condition: " << (res ? "matched" : "mismatched") << ". Fields:" << Endl;
            if (res) {
                for (const auto& pair : map) {
                    Cout << "key='" << pair.first << "'. Value='" << pair.second << "'" << Endl;
                }
            }
            Cout << Endl;
        }
    }

    class TModeArg: public TMainClass {
        int operator()(const int argc, const char** argv) override {
            TString condition;

            NLastGetopt::TOpts opts;
            opts.AddHelpOption();
            opts.AddCharOption('c', "condition").StoreResult(&condition).Required();
            opts.SetFreeArgsNum(0);
            NLastGetopt::TOptsParseResult(&opts, argc, argv);

            Run(condition);

            return 0;
        }
    };

    class TModeConfig: public TMainClass {
        int operator()(const int argc, const char** argv) override {
            TString configPath;
            TString conditionXpath;

            NLastGetopt::TOpts opts;
            opts.AddHelpOption();
            opts.AddLongOption('c', "config", "file path").StoreResult(&configPath).Required();
            opts.AddLongOption("condition_set", "xpath to 'condition_set'")
                .StoreResult(&conditionXpath)
                .Required();
            opts.SetFreeArgsNum(0);
            NLastGetopt::TOptsParseResult res(&opts, argc, argv);

            const NXml::TConfig config = NXml::TConfig::ReadFromFile(configPath);
            NCfg::ValidateConfig(config);

            Run(config.AsString(conditionXpath + "/condition"));

            return 0;
        }
    };

    int operator()(const int argc, const char** argv) override {
        TModChooser chooser;
        chooser.SetDescription("Check tskv condition: from stdin\n");

        auto t = MakeHolder<TModeArg>();
        auto c = MakeHolder<TModeConfig>();

        chooser.AddMode("arg", t.Get(), "get condition from CLI arg");
        chooser.AddMode("config", c.Get(), "get condition from config");

        return chooser.Run(argc, argv);
    }
};

class TModeTsvToFields: public TMainClass {
    static void Process(const NTsv::IParsingPolicy& policy) {
        TString tmp;
        while (Cin.ReadLine(tmp) > 0) {
            NTsv::TLine line;
            Cout << "Line: '" << tmp << "'" << Endl;
            Cout << "Parsing succeed: " << (policy.ParseLine(tmp, line) ? "yes" : "no") << ". Fields:" << Endl;

            for (size_t idx = 0; idx < line.size(); ++idx) {
                Cout << "Field#" << idx + 1 << "='" << line.at(idx) << "'" << Endl;
            }
            Cout << Endl;
        }
    }

    int operator()(const int argc, const char** argv) override {
        TString configPath;
        TString parserXpath;

        NLastGetopt::TOpts opts;
        opts.AddHelpOption();
        opts.AddLongOption('c', "config", "file path").StoreResult(&configPath).Required();
        opts.AddLongOption('p', "parser", "xpath to 'parser'")
            .StoreResult(&parserXpath)
            .Required();
        opts.SetFreeArgsNum(0);
        NLastGetopt::TOptsParseResult res(&opts, argc, argv);

        const NXml::TConfig config = NXml::TConfig::ReadFromFile(configPath);
        NCfg::ValidateConfig(config);
        NTsv::TParser::TArgs args = NCfg::GetTsvParserArgs(config, NCfgList::GetTsvParser(config, parserXpath));

        Process(*args.Policy);

        return 0;
    }
};

class TModeRunParser: public TMainClass {
    static void Process(TParserBase& parser) {
        TString tmp;
        while (Cin.ReadLine(tmp) > 0) {
            Cout << "'" << tmp << "'" << Endl;
            Cout << "Parsing succeed: " << (parser.ProcessLine(tmp) ? "yes" : "no") << Endl;
            parser.Flush();

            TString unistat;
            {
                NUnistat::TBuilder builder(unistat);
                parser.AddUnistat(builder);
            }
            Cout << unistat << Endl;
            Cout << Endl;
        }
    }

    class TModeTsv: public TMainClass {
        int operator()(const int argc, const char** argv) override {
            TString configPath;
            TString parserXpath;

            NLastGetopt::TOpts opts;
            opts.AddHelpOption();
            opts.AddLongOption('c', "config", "file path").StoreResult(&configPath).Required();
            opts.AddLongOption('p', "parser", "xpath to 'parser'")
                .StoreResult(&parserXpath)
                .Required();
            opts.SetFreeArgDefaultTitle("signals", "xpath to signal or signal_template");
            NLastGetopt::TOptsParseResult res(&opts, argc, argv);

            const NXml::TConfig config = NXml::TConfig::ReadFromFile(configPath);
            NCfg::ValidateConfig(config);
            parserXpath = NCfg::PrettifyXpath(config, parserXpath);

            NCfgList::TTsvParser t = NCfgList::GetTsvParser(config, parserXpath);

            std::vector<TString> signals = res.GetFreeArgs();
            if (!signals.empty()) {
                t.Signals.clear();
                t.SignalTemplates.clear();
                for (const TString& s : signals) {
                    if (s.Contains("/signal_template")) {
                        t.SignalTemplates.push_back(s);
                    } else {
                        t.Signals.push_back(s);
                    }
                }
            }

            NTsv::TTsvParser parser({"/dev/null", 1}, NCfg::GetTsvParserArgs(config, t));
            Process(parser);

            return 0;
        }
    };

    class TModeTskv: public TMainClass {
        int operator()(const int argc, const char** argv) override {
            TString configPath;
            TString parserXpath;
            TString conditionXpath;

            NLastGetopt::TOpts opts;
            opts.AddHelpOption();
            opts.AddLongOption('c', "config", "file path").StoreResult(&configPath).Required();
            opts.AddLongOption('p', "parser", "xpath to 'parser'")
                .StoreResult(&parserXpath)
                .Required();
            opts.AddLongOption("condition_set", "xpath to 'condition_set'")
                .StoreResult(&conditionXpath)
                .Required();
            opts.SetFreeArgDefaultTitle("signals", "xpath to signal or signal_template");
            NLastGetopt::TOptsParseResult res(&opts, argc, argv);

            const NXml::TConfig config = NXml::TConfig::ReadFromFile(configPath);
            NCfg::ValidateConfig(config);
            parserXpath = NCfg::PrettifyXpath(config, parserXpath);

            NCfgList::TTskvParser t = NCfgList::GetTskvParser(config, parserXpath);

            t.Conditions.erase(std::remove_if(t.Conditions.begin(),
                                              t.Conditions.end(),
                                              [conditionXpath](const NCfgList::TTskvParser::TConditionSet& p) {
                                                  return p.Xpath != conditionXpath;
                                              }),
                               t.Conditions.end());
            Y_ENSURE(!t.Conditions.empty(), "ConditionSet not found: " << parserXpath);
            Y_ENSURE(1 == t.Conditions.size(), "ConditionSet found to many times: " << parserXpath << ". " << t.Conditions.size());
            NCfgList::TTskvParser::TConditionSet& c = t.Conditions[0];

            std::vector<TString> signals = res.GetFreeArgs();
            if (!signals.empty()) {
                c.Signals.clear();
                c.SignalTemplates.clear();
                for (const TString& s : signals) {
                    if (s.Contains("/signal_template")) {
                        c.SignalTemplates.push_back(s);
                    } else {
                        c.Signals.push_back(s);
                    }
                }
            }

            NTskv::TTskvParser parser({"/dev/null", 1}, NCfg::GetTskvParserArgs(config, t));
            Process(parser);

            return 0;
        }
    };

    int operator()(const int argc, const char** argv) override {
        TModChooser chooser;
        chooser.SetDescription("Start one parser with some signals: from stdin\n");

        auto t = MakeHolder<TModeTsv>();
        auto k = MakeHolder<TModeTskv>();

        chooser.AddMode("parser", t.Get(), "tsv parser");
        chooser.AddMode("tskv_parser", k.Get(), "tskv parser");

        return chooser.Run(argc, argv);
    }
};

class TModePrintConfig: public TMainClass {
    int operator()(const int argc, const char** argv) override {
        TString configPath;

        NLastGetopt::TOpts opts;
        opts.AddHelpOption();
        opts.AddLongOption('c', "config").StoreResult(&configPath).Required();
        opts.SetFreeArgsNum(0);
        NLastGetopt::TOptsParseResult(&opts, argc, argv);

        const NXml::TConfig config = NXml::TConfig::ReadFromFile(configPath);
        NCfg::ValidateConfig(config);

        Cout << "Printing config: " << configPath << Endl << Endl;

        for (const NCfgList::TTsvParser& parser : NCfgList::ListTsvParsers(config, CONFIG_XPATH)) {
            Cout << parser.Xpath << Endl;
            for (const TString& s : parser.Signals) {
                Cout << "    " << s << Endl;
            }
            for (const TString& s : parser.SignalTemplates) {
                Cout << "    " << s << Endl;
            }
            Cout << Endl;
        }

        for (const NCfgList::TTskvParser& parser : NCfgList::ListTskvParsers(config, CONFIG_XPATH)) {
            Cout << parser.Xpath << Endl;
            for (const NCfgList::TTskvParser::TConditionSet& set : parser.Conditions) {
                Cout << "    " << set.Xpath << Endl;

                for (const TString& s : set.Signals) {
                    Cout << "        " << s << Endl;
                }
                for (const TString& s : set.SignalTemplates) {
                    Cout << "        " << s << Endl;
                }
                Cout << Endl;
            }
            Cout << Endl;
        }

        return 0;
    }
};

class TModeValidateConfig: public TMainClass {
    int operator()(const int argc, const char** argv) override {
        TString configPath;

        NLastGetopt::TOpts opts;
        opts.AddHelpOption();
        opts.AddLongOption('c', "config").StoreResult(&configPath).Required();
        opts.SetFreeArgsNum(0);
        NLastGetopt::TOptsParseResult(&opts, argc, argv);

        const NXml::TConfig config = NXml::TConfig::ReadFromFile(configPath);
        Y_ENSURE(NCfg::ValidateConfig(config), "Validation failed");

        return 0;
    }
};

class TModeXmlEscaping: public TMainClass {
    int operator()(const int argc, const char** argv) override {
        NLastGetopt::TOpts opts;
        opts.AddHelpOption();
        opts.SetFreeArgsNum(1);
        opts.SetFreeArgTitle(0, "string_to_escape");
        NLastGetopt::TOptsParseResult res(&opts, argc, argv);

        Cout << EncodeXML(res.GetFreeArgs()[0].c_str()) << Endl;

        return 0;
    }
};

int main(int argc, const char** argv) {
    try {
        TModChooser chooser;
        chooser.SetDescription(
            "Tool for debugging and testing with xunistater\n"
            "Site: https://wiki.yandex-team.ru/passport/xunistater\n"
            "\n");

        auto t = MakeHolder<TModeTskvCondition>();
        auto l = MakeHolder<TModeTsvToFields>();
        auto p = MakeHolder<TModePrintConfig>();
        auto v = MakeHolder<TModeValidateConfig>();
        auto r = MakeHolder<TModeRunParser>();
        auto e = MakeHolder<TModeXmlEscaping>();

        chooser.AddMode("tskv_condition", t.Get(), "parsing tskv-line and applying condition");
        chooser.AddMode("tsv_to_fields", l.Get(), "parsing tsv-line to fields");
        chooser.AddMode("print_config", p.Get(), "print xpathes from config");
        chooser.AddMode("validate_config", v.Get(), "validate config with DTD-schema - in soft mode");
        chooser.AddMode("run_parser", r.Get(), "run one parser with signals");
        chooser.AddMode("xml_escaping", e.Get(), "xml-escaping of string - for regex");

        return chooser.Run(argc, argv);
    } catch (const std::exception& e) {
        Cerr << "Fatal error in main(): " << e.what() << Endl;
        return 1;
    }
}
