#include "parsers.h"

#include "list.h"
#include "signal_templates.h"
#include "signals.h"

#include <passport/infra/libs/cpp/xml/config.h>

#include <unordered_map>

namespace NPassport::NXunistater::NCfg {
    static const std::unordered_map<TString, decltype(&InitTsvSignalRps)> TSV_COLUMN_FACTORIES = {
        {"hgram", InitTsvSignalHistogram},
        {"regex", InitTsvSignalRegex},
        {"rps", InitTsvSignalRps},
        {"string_set", InitTsvSignalStringSet},
        {"substring", InitTsvSignalSubString},
        {"sum", InitTsvSignalSum},
    };

    static const std::unordered_map<TString, decltype(&InitSignalRps)> TSKV_COLUMN_FACTORIES = {
        {"hgram", InitSignalHistogram},
        {"regex", InitSignalRegex},
        {"rps", InitSignalRps},
        {"string_set", InitSignalStringSet},
        {"substring", InitSignalSubString},
        {"sum", InitSignalSum},
    };

    static NTsv::TParsingPolicy GetTsvPolicy(const NXml::TConfig& config, const TString& xpath) {
        const bool hasDelimiter = config.Contains(xpath + "/delimeter_ascii_number");
        const bool hasRegex = config.Contains(xpath + "/regex");
        Y_ENSURE(!hasDelimiter || !hasRegex,
                 "Invalid config for parser: " << xpath << " has /regex and /delimeter_ascii_number: " << xpath);

        if (hasRegex) {
            const TString regex = config.AsString(xpath + "/regex");
            TLog::Info() << xpath << " uses regex: " << regex;
            return NTsv::TRegexingPolicy::Create(TString(regex));
        }

        const auto delimeter = config.AsNum<NTsv::TSplittingPolicy::TDelimeter>(xpath + "/delimeter_ascii_number", ' ');
        TLog::Info() << xpath << " uses spliting with symbol: '" << char(delimeter) << "'";
        return NTsv::TSplittingPolicy::Create(delimeter);
    }

    TParserBase::TBaseSettings GetBaseParserSettings(const NXml::TConfig& config, const TString& xpath) {
        TParserBase::TBaseSettings settings;

        settings.Filename = config.AsString(xpath + "/@file_name");
        settings.LinesBeforeFlush = config.AsInt(xpath + "/lines_before_flush", 1024);
        settings.MissingPolicy =
            config.AsBool(xpath + "/@missingok", false)
                ? TParserBase::TBaseSettings::EMissing::Ok
                : TParserBase::TBaseSettings::EMissing::Fail;
        settings.IgnoreParsingErrors =
            config.AsBool(xpath + "/ignore_parsing_errors", false)
                ? TParserBase::TBaseSettings::EIgnoreParsingErrors::True
                : TParserBase::TBaseSettings::EIgnoreParsingErrors::False;
        settings.IgnoreProcessingErrors =
            config.AsBool(xpath + "/ignore_processing_errors", false)
                ? TParserBase::TBaseSettings::EIgnoreProcessingErrors::True
                : TParserBase::TBaseSettings::EIgnoreProcessingErrors::False;

        return settings;
    }

    NTsv::TParser::TArgs GetTsvParserArgs(const NXml::TConfig& config, const NCfgList::TTsvParser& xpath) {
        NTsv::TParser::TArgs args;
        for (const TString& s : xpath.Signals) {
            const TString type = config.AsString(s + "/@type");

            auto it = TSV_COLUMN_FACTORIES.find(type);
            Y_ENSURE(it != TSV_COLUMN_FACTORIES.end(), "unknown type '" << type << "' in " << s;);
            args.ColumnProcessors.push_back(it->second(config, s));
        }

        for (const TString& s : xpath.SignalTemplates) {
            args.SignalTemplates.push_back(CreateSignalTemplateForTsv(config, s));
        }

        args.Policy = GetTsvPolicy(config, xpath.Xpath);

        return args;
    }

    NTskv::TParser::TArgs GetTskvParserArgs(const NXml::TConfig& config, const NCfgList::TTskvParser& xpath) {
        NTskv::TParser::TArgs args;

        for (const NCfgList::TTskvParser::TConditionSet& conditionSet : xpath.Conditions) {
            NTskv::TConditionSet cond;
            const TString c = NTskv::NConds::TrimExpression(config.AsString(conditionSet.Xpath + "/condition", ""));
            TLog::Info() << "Tskv parser (" << conditionSet.Xpath << ") uses condition: '" << c << "'";
            try {
                NTskv::NConds::ValidateExpression(c);
                cond.Condition = NTskv::NConds::ExpressionToCondition(c);
            } catch (const std::exception& e) {
                TLog::Error() << "Failed to parse condition: " << e.what();
                throw;
            }

            for (const TString& signal : conditionSet.Signals) {
                const TString type = config.AsString(signal + "/@type");
                const TString key = config.AsString(signal + "/key");

                auto it = TSKV_COLUMN_FACTORIES.find(type);
                Y_ENSURE(it != TSKV_COLUMN_FACTORIES.end(), "unknown type '" << type << "' in " << signal);

                cond.Signals.emplace(key, it->second(config, signal));
            }

            for (const TString& s : conditionSet.SignalTemplates) {
                cond.SignalTemplates.push_back(NCfg::CreateSignalTemplateForTskv(config, s));
            }

            args.Conditions.push_back(std::move(cond));
        }

        return args;
    }
}
