#include "signal_templates.h"

#include "signals.h"

#include <passport/infra/libs/cpp/utils/string/string_utils.h>
#include <passport/infra/libs/cpp/xml/config.h>

#include <util/generic/set.h>

namespace NPassport::NXunistater::NCfg {
    NSt::TSignalTemplateCtx CreateSignalTemplateForTsv(const NXml::TConfig& config, const TString& xpath) {
        return CreateSignalTemplate(config, xpath, CreateFieldGetterTsvName, CreateFieldGetterTsvValue);
    }

    NSt::TSignalTemplateCtx CreateSignalTemplateForTskv(const NXml::TConfig& config, const TString& xpath) {
        return CreateSignalTemplate(config, xpath, CreateFieldGetterTskvName, CreateFieldGetterTskvValue);
    }

    static NSt::TTokenGetter::TFuncPtr CreateTokenFunc(const NXml::TConfig& config, const TString& xpath) {
        const TString funcName = config.AsString(xpath + "/@func");

        if ("as_is" == funcName) {
            return NSt::TTokenGetter::TFuncAsIs::Create();
        }

        if ("regex_group" == funcName) {
            return NSt::TTokenGetter::TFuncRegexGroup::Create(
                config.AsString(xpath + "/in"),
                config.AsString(xpath + "/out"));
        }

        ythrow yexception() << "func='" << funcName << "' is not supported: " << xpath;
    }

    static NSt::TTokenPostProcessor CreateTokenPostProcMap(const NXml::TConfig& config, const TString& xpath) {
        NSt::TTokenPostProcMap::TMap map;

        for (const TString& path : config.SubKeys(xpath + "/pair")) {
            TString from;
            try {
                from = config.AsString(path + "/@from");
            } catch (const NXml::TConfig::TEmptyException&) {
                // it is ok for 'from' - for example: user tries to map empty value to 'unknown'
            }

            const TString to = config.AsString(path + "/@to");

            Y_ENSURE(map.emplace(from, to).second,
                     "config contains duplicated 'from': " << xpath);
        }

        std::optional<TString> defVal;
        if (config.Contains(xpath + "/default")) {
            defVal = config.AsString(xpath + "/default");
        }

        return NSt::TTokenPostProcMap::Create(std::move(map), std::move(defVal));
    }

    static NSt::TTokenPostProcessor CreateTokenPostProc(const NXml::TConfig& config, const TString& xpath) {
        const TString funcName = config.AsString(xpath + "/@func");

        if ("map" == funcName) {
            return CreateTokenPostProcMap(config, xpath);
        }

        ythrow yexception() << "func='" << funcName << "' is not supported: " << xpath;
    }

    static NSt::TTokenPostProcessors CreateTokenPostProcs(const NXml::TConfig& config, const TString& xpath) {
        NSt::TTokenPostProcessors res;

        for (const TString& p : config.SubKeys(xpath + "/post_process")) {
            res.push_back(CreateTokenPostProc(config, p));
        }

        return res;
    }

    static std::vector<TString> CreatePersistent(const NXml::TConfig& config,
                                                 const TString& xpath) {
        std::vector<TString> res;

        for (const TString& p : config.SubKeys(xpath + "/persistent_value")) {
            res.push_back(config.AsString(p));
        }

        return res;
    }

    static NSt::TNameBuilder CreateNameBuilder(const NXml::TConfig& config,
                                               const TString& xpath,
                                               TFGCreator fgCreatorName) {
        std::vector<NSt::TTokenGetter> tokens;
        for (const TString& path : config.SubKeys(xpath + "/name/token")) {
            NSt::TTokenGetter::TArgs args{
                .Getter = fgCreatorName(config, path),
                .Func = CreateTokenFunc(config, path),
                .Id = config.AsString(path + "/@id"),
                .PostProcs = CreateTokenPostProcs(config, path),
                .PersistentValues = CreatePersistent(config, path),
            };
            tokens.emplace_back(std::move(args));
        }

        return NSt::TNameBuilder(std::move(tokens), config.AsString(xpath + "/name/pattern"));
    }

    NSt::TSignalTemplateCtx CreateSignalTemplate(const NXml::TConfig& config,
                                                 const TString& xpath,
                                                 TFGCreator fgCreatorName,
                                                 TFGCreator fgCreatorValue) {
        return {
            std::make_unique<NSt::TSignalTemplate>(
                CreateNameBuilder(config, xpath, fgCreatorName),
                fgCreatorValue(config, xpath + "/value"),
                CreateFactory(config, xpath + "/value")),
            config.AsBool(xpath + "/@ignore_errors", false)
                ? NSt::TSignalTemplateCtx::EIgnoreErrors::True
                : NSt::TSignalTemplateCtx::EIgnoreErrors::False,
        };
    }

    NSt::TFieldGetter CreateFieldGetterTsvName(const NXml::TConfig& config, const TString& xpath) {
        const ui32 column = config.AsNum<ui32>(xpath + "/column");
        Y_ENSURE(column > 0, "column begins from 1: " << xpath);

        NSt::TFieldGetter::TArraySettings s;
        s.Column = column - 1;
        return NSt::TFieldGetter(s);
    }

    NSt::TFieldGetter CreateFieldGetterTskvName(const NXml::TConfig& config, const TString& xpath) {
        const TString from = config.AsString(xpath + "/@from");
        Y_ENSURE("value" == from, "Only 'value' is supported for '@from': " << xpath);

        NSt::TFieldGetter::TKvSettings s;
        s.Key = config.AsString(xpath + "/key");
        s.Missingok = config.AsBool(xpath + "/key/@missingok", false);

        return NSt::TFieldGetter(s);
    }

    NSt::TFieldGetter CreateFieldGetterTsvValue(const NXml::TConfig& config, const TString& xpath) {
        return config.AsString(xpath + "/@type") == "rps"
                   ? NSt::TFieldGetter(NSt::TFieldGetter::TArraySettings())
                   : CreateFieldGetterTsvName(config, xpath);
    }

    NSt::TFieldGetter CreateFieldGetterTskvValue(const NXml::TConfig& config, const TString& xpath) {
        NSt::TFieldGetter::TKvSettings s;
        s.Key = config.AsString(xpath + "/key");
        return NSt::TFieldGetter(s);
    }

    NSt::TSignalFactoryPtr CreateFactory(const NXml::TConfig& config, const TString& xpath) {
        const TString type = config.AsString(xpath + "/@type");

        if (type == "hgram") {
            THgramSettings s = ParseHistogramSettings(config, xpath);
            return std::make_unique<NSt::THgramFactory>(std::move(s));
        }

        if (type == "regex") {
            TRegexSettings s = ParseRegexSettings(config, xpath);
            return std::make_unique<NSt::TRegexFactory>(s.Suffix, s.Regex);
        }

        if (type == "rps") {
            TRpsSettings s = ParseRpsSettings(config, xpath);
            return std::make_unique<NSt::TRpsFactory>(s.Suffix);
        }

        if (type == "substring") {
            TSubstringSettings s = ParseSubStringSettings(config, xpath);
            return std::make_unique<NSt::TSubstringFactory>(s);
        }

        if (type == "sum") {
            TSumSettings s = ParseSumSettings(config, xpath);
            return std::make_unique<NSt::TSumFactory>(s.Suffix);
        }

        ythrow yexception() << "type=='" << type << "' is not supported: " << xpath;
    }
}
