#include "signals.h"

#include <passport/infra/daemons/xunistater/src/signal_providers/string_set.h>

#include <passport/infra/libs/cpp/utils/string/string_utils.h>
#include <passport/infra/libs/cpp/xml/config.h>

namespace NPassport::NXunistater::NCfg {
    THgramSettings ParseHistogramSettings(const NXml::TConfig& config, const TString& xpath) {
        THgramSettings res;

        res.Bounds = TSignalHistogram::ParseBounds(
            config.AsString(xpath + "/bounds", TSignalHistogram::GetDefaultBounds()));

        res.AllowManyValues = config.AsBool(xpath + "/allow_many_values", false)
                                  ? TSignalHistogram::EAllowManyValues::True
                                  : TSignalHistogram::EAllowManyValues::False;

        res.Multiplier = config.AsNum<ui64>(xpath + "/multiplier", res.Multiplier);

        return res;
    }

    TRegexSettings ParseRegexSettings(const NXml::TConfig& config, const TString& xpath) {
        TRegexSettings res;
        res.Suffix = config.AsString(xpath + "/suffix", DEFAULT_SUFFIX);
        res.Regex = config.AsString(xpath + "/regex");
        return res;
    }

    TRpsSettings ParseRpsSettings(const NXml::TConfig& config, const TString& xpath) {
        TRpsSettings res;
        res.Suffix = config.AsString(xpath + "/suffix", DEFAULT_SUFFIX);
        return res;
    }

    TSubstringSettings ParseSubStringSettings(const NXml::TConfig& config, const TString& xpath) {
        TSubstringSettings res;
        res.Suffix = config.AsString(xpath + "/suffix", DEFAULT_SUFFIX);
        res.Value = config.AsString(xpath + "/value");
        res.CountOccurrences = config.AsBool(xpath + "/count_occurrences", false)
                                   ? TSignalSubstring::ECountOccurrences::True
                                   : TSignalSubstring::ECountOccurrences::False;
        return res;
    }

    TSumSettings ParseSumSettings(const NXml::TConfig& config, const TString& xpath) {
        TSumSettings res;
        res.Suffix = config.AsString(xpath + "/suffix", DEFAULT_SUFFIX);
        return res;
    }

    TSignal InitSignalHistogram(const NXml::TConfig& config, const TString& xpath) {
        const TString name = config.AsString(xpath + "/name");

        THgramSettings s = ParseHistogramSettings(config, xpath);
        return std::make_unique<TSignalHistogram>(name, std::move(s));
    }

    TSignal InitSignalRegex(const NXml::TConfig& config, const TString& xpath) {
        const TString name = config.AsString(xpath + "/name");
        TRegexSettings s = ParseRegexSettings(config, xpath);

        return std::make_unique<TSignalRegex>(name, s.Suffix, s.Regex);
    }

    TSignal InitSignalRps(const NXml::TConfig& config, const TString& xpath) {
        const TString name = config.AsString(xpath + "/name");
        TRpsSettings s = ParseRpsSettings(config, xpath);

        return std::make_unique<TSignalRps>(name, s.Suffix);
    }

    TSignal InitSignalStringSet(const NXml::TConfig& config, const TString& xpath) {
        const TString prefix = config.AsString(xpath + "/prefix", "");
        const TString pattern = config.AsString(xpath + "/pattern", "");
        Y_ENSURE((prefix || pattern) && (prefix.empty() != pattern.empty()),
                 xpath << ". There must be 'prefix' OR 'pattern'");

        const TString suffix = config.AsString(xpath + "/suffix", DEFAULT_SUFFIX);

        const bool unqote = config.AsBool(xpath + "/unquote", false);

        TSignalStringSet::TSettings settings;
        settings.Prefix = prefix;
        settings.Pattern = pattern;
        settings.Unqote = unqote ? TSignalStringSet::TSettings::EUnquote::True
                                 : TSignalStringSet::TSettings::EUnquote::False;

        if (config.Contains(xpath + "/regex_filter")) {
            const TString regexFilter = config.AsString(xpath + "/regex_filter/expression");
            const bool regexSelect = config.AsBool(xpath + "/regex_filter/select_matched", true);

            settings.RegexFilter = regexFilter;
            settings.RegexSelect = regexSelect ? TSignalStringSet::TSettings::ERegexSelect::Matched
                                               : TSignalStringSet::TSettings::ERegexSelect::NonMatched;
        } else {
            Y_ENSURE(!pattern, "'pattern' without 'regex_tilter' has no sense");
        }

        for (const TString& p : config.SubKeys(xpath + "/persistent_name")) {
            settings.PersistentValues.insert(config.AsString(p));
        }

        return std::make_unique<TSignalStringSet>(suffix, settings);
    }

    TSignal InitSignalSubString(const NXml::TConfig& config, const TString& xpath) {
        const TString name = config.AsString(xpath + "/name");
        TSubstringSettings s = ParseSubStringSettings(config, xpath);

        return std::make_unique<TSignalSubstring>(name, s);
    }

    TSignal InitSignalSum(const NXml::TConfig& config, const TString& xpath) {
        const TString name = config.AsString(xpath + "/name");
        TSumSettings s = ParseSumSettings(config, xpath);

        return std::make_unique<TSignalSum>(name, s.Suffix);
    }

    NTsv::TColumnProcessor InitTsvSignalHistogram(const NXml::TConfig& config, const TString& xpath) {
        const int column = config.AsInt(xpath + "/column");

        return {NTsv::TColumnGetter(column), InitSignalHistogram(config, xpath)};
    }

    NTsv::TColumnProcessor InitTsvSignalRegex(const NXml::TConfig& config, const TString& xpath) {
        std::optional<ui32> column;
        if (config.Contains(xpath + "/column")) {
            column = config.AsInt(xpath + "/column");
        }

        return {NTsv::TColumnGetter(column), InitSignalRegex(config, xpath)};
    }

    NTsv::TColumnProcessor InitTsvSignalRps(const NXml::TConfig& config, const TString& xpath) {
        return {NTsv::TColumnGetter({}), InitSignalRps(config, xpath)};
    }

    NTsv::TColumnProcessor InitTsvSignalStringSet(const NXml::TConfig& config, const TString& xpath) {
        const int column = config.AsInt(xpath + "/column");

        return {NTsv::TColumnGetter(column), InitSignalStringSet(config, xpath)};
    }

    NTsv::TColumnProcessor InitTsvSignalSubString(const NXml::TConfig& config, const TString& xpath) {
        std::optional<ui32> column;
        if (config.Contains(xpath + "/column")) {
            column = config.AsInt(xpath + "/column");
        }

        return {NTsv::TColumnGetter(column), InitSignalSubString(config, xpath)};
    }

    NTsv::TColumnProcessor InitTsvSignalSum(const NXml::TConfig& config, const TString& xpath) {
        const int column = config.AsInt(xpath + "/column");

        return {NTsv::TColumnGetter(column), InitSignalSum(config, xpath)};
    }
}
