#include "parser_base.h"

#include <util/system/fs.h>
#include <util/system/fstat.h>

namespace NPassport::NXunistater {
    static const TString LINE_BREAK('\n');

    TParserBase::TParserBase(const TBaseSettings& settings)
        : Settings_(settings)
        , File_(Openfile(Settings_.Filename, Settings_.MissingPolicy))
    {
    }

    ui64 TParserBase::GetErrors() const {
        return ParsingErrors_.GetValue();
    }

    void TParserBase::Run() {
        {
            TFileDetails fd = TryReopenfile(Settings_.Filename, Settings_.MissingPolicy, File_.Inode);
            if (fd.Input) {
                File_ = std::move(fd);
            }
        }
        if (!File_.Input) {
            return;
        }

        ui64 idx = 0;

        std::optional<NTail::TLine> data;
        while ((data = File_.Input->ReadLine())) {
            data->Line.ChopSuffix(LINE_BREAK);

            if (!ProcessLine(data->Line)) {
                ++ParsingErrors_;
            }

            if (++idx > Settings_.LinesBeforeFlush) {
                idx = 0;
                Flush();
                ParsingErrors_.Flush();
            }
        }

        Flush();
        ParsingErrors_.Flush();
    }

    void TParserBase::LogError(TStringBuf valueId, TStringBuf line, TStringBuf err) const {
        if (Settings_.IgnoreProcessingErrors == TBaseSettings::EIgnoreProcessingErrors::True) {
            return;
        }

        ++ParsingErrors_;
        TLog::Error() << Settings_.Filename
                      << ". Failed to process line: '" << line
                      << "'. " << valueId
                      << ". Error: " << err;
    }

    void TParserBase::LogException(TStringBuf valueId, TStringBuf line, TStringBuf err) const {
        ++ParsingErrors_;
        TLog::Error() << Settings_.Filename
                      << ". Failed to process line: '" << line
                      << "'. " << valueId
                      << ". Exception: " << err;
    }

    TParserBase::TFileDetails TParserBase::Openfile(const TString& filename,
                                                    TBaseSettings::EMissing missingPolicy) {
        if (!NFs::Exists(filename) && TBaseSettings::EMissing::Ok == missingPolicy) {
            return {};
        }

        TFile file(filename, OpenExisting | RdOnly | Seq);
        const TFileStat fs(file);

        std::unique_ptr<NTail::TTail> res = std::make_unique<NTail::TTail>(file);
        size_t size = res->SkipAll();
        TLog::Info() << "File " << filename << " opened. Skipped " << size << " bytes";

        return {std::move(res), fs.INode};
    }

    TParserBase::TFileDetails TParserBase::TryReopenfile(const TString& filename,
                                                         TBaseSettings::EMissing missingPolicy,
                                                         ui64 inode) {
        if (!NFs::Exists(filename) && TBaseSettings::EMissing::Ok == missingPolicy) {
            return {};
        }

        TFile file(filename, OpenExisting | RdOnly | Seq);
        const TFileStat fs(file);
        if (inode == fs.INode) {
            return {};
        }

        std::unique_ptr<NTail::TTail> res = std::make_unique<NTail::TTail>(file);
        TLog::Info() << "File " << filename << " opened: inode changed";

        return {std::move(res), fs.INode};
    }
}
