#include "tskv.h"

#include <library/cpp/string_utils/tskv_format/tskv_map.h>

namespace NPassport::NXunistater::NTskv {
    TTskvParser::TTskvParser(const TBaseSettings& settings, TArgs&& args)
        : TParserBase(settings)
        , Conditions_(std::move(args.Conditions))
    {
    }

    void TTskvParser::AddUnistat(NUnistat::TBuilder& builder) const {
        for (const TConditionSet& cond : Conditions_) {
            for (const auto& pair : cond.Signals) {
                pair.second->AddUnistat(builder);
            }
            for (const NSt::TSignalTemplateCtx& cs : cond.SignalTemplates) {
                cs.Templ->AddUnistat(builder);
            }
        }
    }

    bool TTskvParser::ProcessLine(TStringBuf line) {
        Buf_.clear();
        Map_.clear();

        TErrorMsg err = ParseLine(line, Map_, Buf_);
        if (err) {
            if (Settings_.IgnoreParsingErrors == TBaseSettings::EIgnoreParsingErrors::True) {
                return true;
            }

            TLog::Error() << Settings_.Filename << ". Failed to parse tskv: '" << line << "'. " << err;
            return false;
        }

        for (TConditionSet& cond : Conditions_) {
            ProccessCondition(Map_, cond, line);
        }

        return true;
    }

    void TTskvParser::Flush() {
        for (TConditionSet& cond : Conditions_) {
            for (auto& pair : cond.Signals) {
                pair.second->FlushBuffer();
            }
            for (NSt::TSignalTemplateCtx& cs : cond.SignalTemplates) {
                cs.Templ->FlushBuffer();
            }
        }
    }

    TErrorMsg TTskvParser::ParseLine(TStringBuf line, TKeyValue& map, TString& buf) {
        line.SkipPrefix("tskv\t");

        try {
            NTskvFormat::DeserializeMap(line, map, buf);
        } catch (const std::exception& e) {
            return e.what();
        }

        return {};
    }

    void TTskvParser::ProccessCondition(const TKeyValue& values, TConditionSet& set, TStringBuf line) const {
        if (!set.Condition->Match(values)) {
            return;
        }

        for (auto& pair : set.Signals) {
            auto it = values.find(pair.first);
            if (it == values.end()) {
                continue;
            }

            try {
                TErrorMsg err = pair.second->Process(it->second);
                if (err) {
                    LogError(pair.first, line, err);
                }
            } catch (const std::exception& e) {
                LogException(pair.first, line, e.what());
            }
        }

        for (NSt::TSignalTemplateCtx& cs : set.SignalTemplates) {
            try {
                TErrorMsg err = cs.Templ->Process(values);
                if (err && cs.IgnoreErrors == NSt::TSignalTemplateCtx::EIgnoreErrors::False) {
                    LogError("", line, err);
                }
            } catch (const std::exception& e) {
                LogException("", line, e.what());
            }
        }
    }
}
