#include "tskv_condition.h"

#include <passport/infra/libs/cpp/utils/string/format.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <util/string/subst.h>

namespace NPassport::NXunistater::NTskv::NConds {
    TString TrimExpression(TStringBuf expression) {
        TString str(expression);
        SubstGlobal(str, "\n", "");
        SubstGlobal(str, ")", " ) ");
        SubstGlobal(str, "(", " ( ");

        expression = str;
        NUtils::Trim(expression);
        str = expression;

        TString prev;
        do {
            prev = str;
            str = SubstGlobalCopy(str, "  ", " ");
        } while (prev != str);

        SubstGlobal(str, " =", "=");
        SubstGlobal(str, "= ", "=");
        SubstGlobal(str, "! ", "!");
        SubstGlobal(str, " !=", "!=");

        return str;
    }

    void ValidateEqualCount(TStringBuf expression) {
        TStringBuf left;
        TStringBuf restPart;

        while (expression.TrySplit('=', left, restPart)) {
            if (left.EndsWith("!")) {
                Y_ENSURE(left.size() > 1, "Expression can't starts with '!='");
                Y_ENSURE(restPart, "Expression can't ends with '!='");
                Y_ENSURE('=' != restPart.front(), "Expression can contain not-equals only like this '!='");
                Y_ENSURE('!' != restPart.front(), "Expression can contain not-equals only like this '!='");
            } else {
                Y_ENSURE(left, "Expression can't starts with '='");
                Y_ENSURE(restPart, "Expression can't ends with '='");
                Y_ENSURE(restPart.SkipPrefix("="), "Expression can contain equals only like this '=='");
                Y_ENSURE(restPart, "Expression can't ends with '=='");
                Y_ENSURE('=' != restPart.front(), "Expression can contain equals only like this '=='");
                Y_ENSURE('!' != restPart.front(), "Expression can contain equals only like this '=='");
            }

            expression = restPart;
        }
    }

    void ValidateBraceCount(TStringBuf expression) {
        int braces = 0;

        for (const char c : expression) {
            if ('(' == c) {
                ++braces;
            } else if (')' == c) {
                --braces;
            }
            Y_ENSURE(braces >= 0, "Opening and closing braces mismatched");
        }

        Y_ENSURE(braces == 0, "Opening and closing braces mismatched");
    }

    void ValidateExpression(TStringBuf expression) {
        ValidateEqualCount(expression);
        ValidateBraceCount(expression);
    }

    TConditionBuilder::TConditionBuilder(TStringBuf exp)
        : OriginalExpression_(exp)
    {
        if (exp.empty()) {
            Result_ = TTrue::Create();
            return;
        }

        CheckBraceOpened("(");

        while (exp) {
            TStringBuf tok = exp.NextTok(' ');

            switch (LastState_) {
                case EState::BraceOpened:
                    if (CheckBraceOpened(tok) || CheckKeyValue(tok) || CheckBoolFunction(tok)) {
                        continue;
                    }
                    ythrow yexception()
                        << GetErrorMessageIllegalExpression("opened brace OR key-value expression OR bool function", exp, tok);
                case EState::KeyValue:
                    if (CheckAndOr(tok) || CheckBraceClosed(tok)) {
                        continue;
                    }
                    ythrow yexception()
                        << GetErrorMessageIllegalExpression("close brace OR and/or", exp, tok);
                case EState::BraceClosed:
                    if (CheckAndOr(tok) || CheckBraceClosed(tok)) {
                        continue;
                    }
                    ythrow yexception()
                        << GetErrorMessageIllegalExpression("close brace OR and/or", exp, tok);
                case EState::And:
                case EState::Or:
                    if (CheckBraceOpened(tok) || CheckKeyValue(tok) || CheckBoolFunction(tok)) {
                        continue;
                    }
                    ythrow yexception()
                        << GetErrorMessageIllegalExpression("opened brace OR key-value expression OR bool function", exp, tok);
            }
        }

        CheckBraceClosed(")");

        Y_ENSURE(!States_.empty(), "Internal error");
        Y_ENSURE(States_.back().Cond, "Internal error");
        Y_ENSURE(States_.back().E == EState::KeyValue, "Internal error: " << States_.back().E);

        Result_ = std::move(States_.back().Cond);
    }

    bool TConditionBuilder::CheckBraceOpened(TStringBuf tok) {
        if ("(" != tok) {
            return false;
        }

        States_.push_back(TState{EState::BraceOpened, nullptr});
        LastState_ = EState::BraceOpened;
        return true;
    }

    bool TConditionBuilder::CheckBraceClosed(TStringBuf tok) {
        if (")" != tok) {
            return false;
        }

        LastState_ = EState::BraceClosed;

        TCondition cond = std::move(States_.back().Cond);
        Y_ENSURE(EState::KeyValue == States_.back().E, "Internal error: " << States_.back().E);
        Y_ENSURE(cond, "Internal error: missing cond");
        States_.pop_back();

        while (EState::BraceOpened != States_.back().E) {
            Y_ENSURE(EState::Or == States_.back().E, "Internal error: " << States_.back().E);
            States_.pop_back();

            Y_ENSURE(!States_.empty(), "Internal error");
            TCondition kv = std::move(States_.back().Cond);
            Y_ENSURE(EState::KeyValue == States_.back().E, "Internal error: " << States_.back().E);
            Y_ENSURE(kv, "Internal error");
            States_.pop_back();

            cond = TOr::Create(std::move(kv), std::move(cond));
        }

        Y_ENSURE(!States_.empty(), "Internal error");
        if (EState::BraceOpened == States_.back().E) {
            States_.pop_back();
        }

        if (!States_.empty() && EState::And == States_.back().E) {
            States_.pop_back();
            Y_ENSURE(!States_.empty(), "Internal error");

            TCondition kv = std::move(States_.back().Cond);
            Y_ENSURE(EState::KeyValue == States_.back().E, "Internal error: " << States_.back().E);
            Y_ENSURE(kv, "Internal error");
            States_.pop_back();

            cond = TAnd::Create(std::move(kv), std::move(cond));
        }

        States_.push_back(TState{EState::KeyValue, std::move(cond)});
        return true;
    }

    bool TConditionBuilder::CheckAndOr(TStringBuf tok) {
        if ("or" == tok) {
            States_.push_back(TState{EState::Or, nullptr});
            LastState_ = EState::Or;
            return true;
        }

        if ("and" == tok) {
            States_.push_back(TState{EState::And, nullptr});
            LastState_ = EState::And;
            return true;
        }

        return false;
    }

    bool TConditionBuilder::CheckKeyValue(TStringBuf tok) {
        if (!tok.Contains("!=") && !tok.Contains("==")) {
            return false;
        }

        const bool isNotEqual = tok.Contains("!=");

        TStringBuf key = tok.NextTok(isNotEqual ? "!=" : "==");
        Y_ENSURE(key, "Illegal expression: " << key);
        Y_ENSURE(!key.Contains('=') || !key.Contains('!'), "Illegal expression");

        TStringBuf value = tok;
        Y_ENSURE(value, "Illegal expression");
        Y_ENSURE(!value.Contains('=') || !value.Contains('!'), "Illegal expression");

        LastState_ = EState::KeyValue;

        TCondition cond = isNotEqual ? TKvNotEquals::Create(TString(key), TString(value))
                                     : TKvEquals::Create(TString(key), TString(value));

        States_.push_back(TState{EState::KeyValue, RollUpLastState(std::move(cond))});
        return true;
    }

    bool TConditionBuilder::CheckBoolFunction(TStringBuf tok) {
        if (tok == "or" || tok == "and") {
            return false;
        }
        if (tok.Contains('=') || tok.Contains(')')) {
            return false;
        }

        LastState_ = EState::KeyValue;

        TCondition cond = tok.SkipPrefix("!") ? TNotContainKey::Create(TString(tok))
                                              : TContainKey::Create(TString(tok));
        States_.push_back(TState{EState::KeyValue, RollUpLastState(std::move(cond))});

        return true;
    }

    TCondition TConditionBuilder::RollUpLastState(TCondition cond) {
        Y_ENSURE(!States_.empty(), "Internal error");
        if (States_.back().E != EState::And) {
            return cond;
        }

        States_.pop_back();
        Y_ENSURE(!States_.empty(), "Internal error");

        TCondition kv = std::move(States_.back().Cond);
        Y_ENSURE(EState::KeyValue == States_.back().E, "Internal error: " << States_.back().E);
        Y_ENSURE(kv, "Internal error");
        States_.pop_back();

        return TAnd::Create(std::move(kv), std::move(cond));
    }

    TString TConditionBuilder::GetErrorMessageIllegalExpression(const TStringBuf msg, TStringBuf exp, TStringBuf tok) const {
        TStringStream s;
        s << "Illegal expression. Allowed: " << msg << Endl
          << "'" << OriginalExpression_ << "'" << Endl
          << TString(OriginalExpression_.size() - exp.size() - tok.size(), ' ') << "^";
        return s.Str();
    }

    TCondition ExpressionToCondition(TStringBuf exp) {
        TConditionBuilder builder(exp);
        return builder.GetResult();
    }

    TString TranslateExpressionForTest(TStringBuf expression) {
        TString res = TrimExpression(expression);
        ValidateExpression(res);
        return ExpressionToCondition(res)->Serialize();
    }
}
