#pragma once

#include <util/generic/hash.h>
#include <util/generic/noncopyable.h>
#include <util/generic/strbuf.h>
#include <util/generic/string.h>

#include <vector>

namespace NPassport::NXunistater::NTskv::NConds {
    using TKeyValue = THashMap<TStringBuf, TStringBuf>;

    class ICondition: TMoveOnly {
    public:
        virtual ~ICondition() = default;
        virtual bool Match(const TKeyValue& values) const = 0;
        virtual TString Serialize() const = 0;
    };
    using TCondition = std::unique_ptr<ICondition>;

    class TContainKey: public ICondition {
    public:
        TContainKey(const TString& key)
            : Key_(key)
        {
        }

        static TCondition Create(const TString& key) {
            return std::make_unique<TContainKey>(key);
        }

        bool Match(const TKeyValue& values) const override {
            return values.contains(Key_);
        }

        TString Serialize() const override {
            return "CONTAIN(" + Key_ + ")";
        }

    protected:
        const TString Key_;
    };

    class TNotContainKey: public TContainKey {
    public:
        TNotContainKey(const TString& key)
            : TContainKey(key)
        {
        }

        static TCondition Create(const TString& key) {
            return std::make_unique<TNotContainKey>(key);
        }

        bool Match(const TKeyValue& values) const override {
            return !TContainKey::Match(values);
        }

        TString Serialize() const override {
            return "NOT_CONTAIN(" + Key_ + ")";
        }
    };

    class TKvEquals: public ICondition {
    public:
        TKvEquals(const TString& key, const TString& value)
            : Key_(key)
            , Value_(value)
        {
        }

        static TCondition Create(const TString& key, const TString& value) {
            return std::make_unique<TKvEquals>(key, value);
        }

        bool Match(const TKeyValue& values) const override {
            auto it = values.find(Key_);
            return it != values.end() && it->second == Value_;
        }

        TString Serialize() const override {
            return Key_ + "==" + Value_;
        }

    protected:
        const TString Key_;
        const TString Value_;
    };

    class TKvNotEquals: public TKvEquals {
    public:
        using TKvEquals::TKvEquals;

        static TCondition Create(const TString& key, const TString& value) {
            return std::make_unique<TKvNotEquals>(key, value);
        }

        bool Match(const TKeyValue& values) const override {
            return !TKvEquals::Match(values);
        }

        TString Serialize() const override {
            return Key_ + "!=" + Value_;
        }
    };

    class TBinary: public ICondition {
    public:
        TBinary(TCondition l, TCondition r)
            : L_(std::move(l))
            , R_(std::move(r))
        {
        }

    protected:
        TCondition L_;
        TCondition R_;
    };

    class TAnd: public TBinary {
    public:
        using TBinary::TBinary;

        static TCondition Create(TCondition l, TCondition r) {
            return std::make_unique<TAnd>(std::move(l), std::move(r));
        }

        bool Match(const TKeyValue& values) const override {
            return L_->Match(values) && R_->Match(values);
        }

        TString Serialize() const override {
            return "(" + L_->Serialize() + " && " + R_->Serialize() + ")";
        }
    };

    class TOr: public TBinary {
    public:
        using TBinary::TBinary;

        static TCondition Create(TCondition l, TCondition r) {
            return std::make_unique<TOr>(std::move(l), std::move(r));
        }

        bool Match(const TKeyValue& values) const override {
            return L_->Match(values) || R_->Match(values);
        }

        TString Serialize() const override {
            return "(" + L_->Serialize() + " || " + R_->Serialize() + ")";
        }
    };

    class TTrue: public ICondition {
    public:
        static TCondition Create() {
            return std::make_unique<TTrue>();
        }

        bool Match(const TKeyValue&) const override {
            return true;
        }

        TString Serialize() const override {
            return "true";
        }
    };

    TString TrimExpression(TStringBuf expression);
    void ValidateExpression(TStringBuf expression);
    TCondition ExpressionToCondition(TStringBuf expression);

    void ValidateEqualCount(TStringBuf expression);
    void ValidateBraceCount(TStringBuf expression);

    TString TranslateExpressionForTest(TStringBuf expression);

    enum class EState {
        BraceOpened,
        BraceClosed,
        KeyValue,
        And,
        Or,
    };

    class TConditionBuilder {
    public:
        TConditionBuilder(TStringBuf exp);

        TCondition GetResult() {
            Y_ENSURE(Result_, "Result moved out already");
            return std::move(Result_);
        }

    protected:
        bool CheckBraceOpened(TStringBuf tok);
        bool CheckBraceClosed(TStringBuf tok);
        bool CheckAndOr(TStringBuf tok);
        bool CheckKeyValue(TStringBuf tok);
        bool CheckBoolFunction(TStringBuf tok);

        TCondition RollUpLastState(TCondition cond);

        TString GetErrorMessageIllegalExpression(const TStringBuf msg, TStringBuf exp, TStringBuf tok) const;

    private:
        struct TState {
            EState E;
            TCondition Cond;
        };

        const TString OriginalExpression_;
        std::vector<TState> States_;
        EState LastState_ = EState::BraceOpened;
        TCondition Result_;
    };
}
