#include "tsv.h"

namespace NPassport::NXunistater::NTsv {
    TColumnGetter::TColumnGetter(std::optional<ui32> column)
        : Column_(column)
        , Str_("full line")
    {
        if (Column_) {
            Y_ENSURE(*Column_ > 0, "columnNo_ must be > 0: " << *Column_);
            Str_ = TStringBuilder() << "field=" << *Column_;
        }
    }

    std::pair<TStringBuf, TErrorMsg> TColumnGetter::GetField(const TLine& lineVec, const TStringBuf lineStr) const {
        if (!Column_) {
            return {lineStr, {}};
        }
        ui32 column = *Column_;

        if (lineVec.size() < column) {
            return {{}, TStringBuilder() << "line has not required " << Str_};
        }

        return {lineVec[column - 1], {}};
    }

    TTsvParser::TTsvParser(const TBaseSettings& settings, TArgs&& args)
        : TParserBase(settings)
        , ColumnProcessors_(std::move(args.ColumnProcessors))
        , SignalTemplates_(std::move(args.SignalTemplates))
        , Policy_(std::move(args.Policy))
    {
    }

    void TTsvParser::AddUnistat(NUnistat::TBuilder& builder) const {
        for (const TColumnProcessor& p : ColumnProcessors_) {
            p.Signal->AddUnistat(builder);
        }
        for (const NSt::TSignalTemplateCtx& cs : SignalTemplates_) {
            cs.Templ->AddUnistat(builder);
        }
    }

    bool TTsvParser::ProcessLine(TStringBuf line) {
        LineVec_.clear();
        if (!Policy_->ParseLine(line, LineVec_)) {
            if (Settings_.IgnoreParsingErrors == TBaseSettings::EIgnoreParsingErrors::True) {
                return true;
            }

            TLog::Error() << Settings_.Filename << ". Failed to parse line: '" << line << "'";
            return false;
        }

        for (TColumnProcessor& p : ColumnProcessors_) {
            try {
                std::pair<TStringBuf, TErrorMsg> pair = p.Getter.GetField(LineVec_, line);
                if (pair.second) {
                    LogError(p.Getter.AsString(), line, pair.second);
                    continue;
                }
                TErrorMsg err = p.Signal->Process(pair.first);
                if (err) {
                    LogError(p.Getter.AsString(), line, err);
                }
            } catch (const std::exception& e) {
                LogException(p.Getter.AsString(), line, e.what());
            }
        }

        for (NSt::TSignalTemplateCtx& cs : SignalTemplates_) {
            try {
                TErrorMsg err = cs.Templ->Process(LineVec_);
                if (err && cs.IgnoreErrors == NSt::TSignalTemplateCtx::EIgnoreErrors::False) {
                    LogError("", line, err);
                }
            } catch (const std::exception& e) {
                LogException("", line, e.what());
            }
        }

        return true;
    }

    void TTsvParser::Flush() {
        for (TColumnProcessor& p : ColumnProcessors_) {
            p.Signal->FlushBuffer();
        }
        for (NSt::TSignalTemplateCtx& cs : SignalTemplates_) {
            cs.Templ->FlushBuffer();
        }
    }

    TSplittingPolicy::TSplittingPolicy(TDelimeter delimeter)
        : Delimeter_(delimeter)
    {
    }

    TParsingPolicy TSplittingPolicy::Create(TDelimeter delimeter) {
        return std::make_unique<TSplittingPolicy>(delimeter);
    }

    bool TSplittingPolicy::ParseLine(TStringBuf lineStr, TLine& lineVec) const {
        TStringBuf buf(lineStr);
        while (buf) {
            lineVec.push_back(buf.NextTok(Delimeter_));
        }
        return true;
    }

    TRegexingPolicy::TRegexingPolicy(const TString& regex)
        : Regex_(regex)
    {
    }

    TParsingPolicy TRegexingPolicy::Create(const TString& regex) {
        return std::make_unique<TRegexingPolicy>(regex);
    }

    bool TRegexingPolicy::ParseLine(TStringBuf lineStr, TLine& lineVec) const {
        return Regex_.FullMatch(RegexCtx_, lineStr, lineVec);
    }
}
