#pragma once

#include "base.h"

#include <passport/infra/libs/cpp/utils/string/split.h>

#include <util/generic/yexception.h>
#include <util/string/cast.h>

#include <array>
#include <optional>
#include <vector>

namespace NPassport::NXunistater {
    class TSignalHistogram: public ISignalProvider {
    public:
        using TBounds = NUnistat::TSignalHistogram<>::TBounds;

        enum class EAllowManyValues {
            True,
            False,
        };

        struct TSettings {
            TBounds Bounds;
            EAllowManyValues AllowManyValues = EAllowManyValues::False;
            ui64 Multiplier = 1000;
        };

        TSignalHistogram(const TString& name, const TSettings& settings)
            : AllowManyValues_(settings.AllowManyValues)
            , Multiplier_(settings.Multiplier)
            , Data_(name, TBounds(settings.Bounds))
        {
        }

        TErrorMsg Process(TStringBuf field) override {
            if (field == "-") {
                return {};
            }

            const std::pair<ui32, TErrorMsg> pair =
                AllowManyValues_ == EAllowManyValues::True ? ManyFieldsToValue(field) : FieldToValue(field);
            if (pair.second) {
                return pair.second;
            }

            Data_.AddValue(pair.first);

            return {};
        }

        void FlushBuffer() noexcept override {
            Data_.Flush();
        }

        void AddUnistat(NUnistat::TBuilder& builder) const override {
            builder.Add(Data_);
        }

    public:
        static TString GetDefaultBounds() {
            return "0,1,2,3,4,5,6,7,8,9,10,12,15,17,20,25,30,35,40,45,50,60,70,80,90,100,125,150,175,200,225,250,275,300,400,500,750,1000,2000,3000";
        }

        static TBounds ParseBounds(const TString& value) {
            TBounds res;

            for (TStringBuf b : NUtils::ToVector<TStringBuf>(value, ',')) {
                NUnistat::TSignalHistogram<>::TValue v = 0;
                Y_ENSURE(TryIntFromString<10>(b, v), "Bounds should be comma-separated numbers: " << value);
                res.push_back(v);
            }

            return res;
        }

    protected:
        std::pair<ui64, TErrorMsg> FieldToValue(TStringBuf field) const {
            if (field == "-") {
                return {0, {}};
            }

            double seconds = 0;
            if (!TryFromString(field, seconds)) {
                return {0, TStringBuilder() << "hgram: unexpected field format - is not double '" << field << "'"};
            }

            return {static_cast<ui64>(seconds * Multiplier_), {}};
        }

        std::pair<ui64, TErrorMsg> ManyFieldsToValue(TStringBuf fields) const {
            if (!fields) {
                return {0, "hgram: field is empty"};
            }

            ui64 res = 0;

            while (fields) {
                TStringBuf f = fields.NextTok(", ");
                while (f) {
                    std::pair<ui64, TErrorMsg> pair = FieldToValue(f.NextTok(" : "));
                    if (pair.second) {
                        return {0, pair.second};
                    }
                    res += pair.first;
                }
            }

            return {res, {}};
        }

    private:
        const EAllowManyValues AllowManyValues_;
        const ui64 Multiplier_;
        NUnistat::TSignalBufferedHistogram Data_;
    };
}
