#pragma once

#include "base.h"

#include <passport/infra/libs/cpp/re2/regex.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <util/generic/hash_set.h>

#include <map>
#include <mutex>
#include <optional>
#include <set>

namespace NPassport::NXunistater {
    class TSignalStringSet: public ISignalProvider {
    public:
        struct TSettings {
            enum class EUnquote {
                True,
                False,
            } Unqote = EUnquote::False;

            std::optional<TString> RegexFilter;
            enum class ERegexSelect {
                Matched,
                NonMatched,
            } RegexSelect = ERegexSelect::Matched;

            std::set<TString> PersistentValues;

            TString Prefix;
            TString Pattern;
        };

        TSignalStringSet(const TString& suffix,
                         const TSettings& settings)
            : Prefix_(settings.Prefix)
            , Suffix_(suffix)
            , Pattern_(NRe2::TPattern::WithClassicPlaceholders(
                  settings.Pattern,
                  GetGroupNumber(settings.RegexFilter)))
            , Unqote_(settings.Unqote)
            , RegexSelectMatched_(settings.RegexSelect == TSettings::ERegexSelect::Matched)
        {
            if (settings.RegexFilter) {
                RegexFilter_.emplace(*settings.RegexFilter);
            }

            for (const TString& p : settings.PersistentValues) {
                Buffer_.Values.emplace(p, 0);
            }
            FlushBuffer();
        }

        TErrorMsg Process(TStringBuf field) override {
            if (Unqote_ == TSettings::EUnquote::True) {
                field.SkipPrefix("\"");
                field.ChopSuffix("\"");
            }

            if (RegexFilter_) {
                const bool matched = RegexFilter_->PartialMatch(RegexCtx_, field, Buffer_.RegexGroups);
                if (matched != RegexSelectMatched_) {
                    return {};
                }
            }

            if (Pattern_.IsEmpty()) {
                ProcPrefix(field);
            } else {
                ProcPattern();
            }

            return {};
        }

        void FlushBuffer() noexcept override {
            {
                std::unique_lock lock(Data_.Mutex);
                for (const auto& pair : Buffer_.Values) {
                    auto [it, inserted] = Data_.Values.emplace(pair.first, TData::TToSerialize{});
                    if (inserted) {
                        it->second.Name = SerializeName(pair.first);
                    }
                    it->second.Val += pair.second;
                }
            }

            for (auto& pair : Buffer_.Values) {
                pair.second = 0;
            }
        }

        void AddUnistat(NUnistat::TBuilder& builder) const override {
            std::unique_lock lock(Data_.Mutex);
            for (const auto& pair : Data_.Values) {
                const TData::TToSerialize& d = pair.second;
                builder.AddRow(d.Name, d.Val);
            }
        }

        using TPattern = std::vector<std::pair<TString, ui32>>;

    protected:
        void ProcPrefix(TStringBuf field) {
            auto it = Buffer_.Values.find(field);
            if (it == Buffer_.Values.end()) {
                Buffer_.Values.emplace(field, 1);
            } else {
                ++it->second;
            }
        }

        void ProcPattern() {
            Buffer_.PatternTmpKey.clear();

            for (const TStringBuf b : Buffer_.RegexGroups) {
                Buffer_.PatternTmpKey.append(b);
                Buffer_.PatternTmpKey.push_back(PATTERN_KEY_DELIM);
            }
            if (Buffer_.PatternTmpKey) {
                Buffer_.PatternTmpKey.pop_back();
            }

            ProcPrefix(Buffer_.PatternTmpKey);
        }

        TString SerializeName(TStringBuf value) const {
            if (Pattern_.IsEmpty()) {
                return NUtils::CreateStr(Prefix_, value, Suffix_);
            }

            const std::vector<TStringBuf> vals = NUtils::ToVectorWithEmpty<TStringBuf>(value, PATTERN_KEY_DELIM);

            TString res = Pattern_.BuildString(vals);
            res.append(Suffix_);

            return res;
        }

        static size_t GetGroupNumber(const std::optional<TString>& regex) {
            return regex ? NRe2::TRegexGroups(*regex).NumberOfGroups() : 0;
        }

    private:
        const TString Prefix_;
        const TString Suffix_;
        const NRe2::TPattern Pattern_;

        const TSettings::EUnquote Unqote_;

        std::optional<NRe2::TRegexGroups> RegexFilter_;
        NRe2::TRegexGroupsCtx RegexCtx_;
        const bool RegexSelectMatched_;

        struct TBuffer {
            THashMap<TString, ui64> Values;
            std::vector<TStringBuf> RegexGroups;
            TString PatternTmpKey;
        } Buffer_;

        struct TData {
            struct TToSerialize {
                ui64 Val = 0;
                TString Name;
            };

            std::map<TString, TToSerialize> Values;
            mutable std::mutex Mutex;
        } Data_;

        static const char PATTERN_KEY_DELIM = 0x00;
    };
}
