#pragma once

#include <passport/infra/daemons/xunistater/src/signal_providers/base.h>
#include <passport/infra/daemons/xunistater/src/signal_providers/histogram.h>
#include <passport/infra/daemons/xunistater/src/signal_providers/regex.h>
#include <passport/infra/daemons/xunistater/src/signal_providers/rps.h>
#include <passport/infra/daemons/xunistater/src/signal_providers/substring.h>
#include <passport/infra/daemons/xunistater/src/signal_providers/sum.h>

namespace NPassport::NXunistater::NSt {
    class ISignalFactory {
    public:
        virtual ~ISignalFactory() = default;
        virtual TSignal Create(const TString& name) const = 0;
    };
    using TSignalFactoryPtr = std::unique_ptr<ISignalFactory>;

    class THgramFactory: public ISignalFactory {
    public:
        THgramFactory(TSignalHistogram::TSettings&& settings)
            : Settings_(std::move(settings))
        {
        }

        TSignal Create(const TString& name) const override {
            // ctor is expensive
            // but there are at most 1000 signals in handle
            return std::make_unique<TSignalHistogram>(name, Settings_);
        }

    private:
        TSignalHistogram::TSettings Settings_;
    };

    class TRegexFactory: public ISignalFactory {
    public:
        TRegexFactory(const TString& suffix, const TString& regex)
            : Suffix_(suffix)
            , Regex_(regex)
        {
        }

        TSignal Create(const TString& name) const override {
            // regex compiling is expensive
            // but there are at most 1000 signals in handle
            return std::make_unique<TSignalRegex>(name, Suffix_, Regex_);
        }

    private:
        TString Suffix_;
        TString Regex_;
    };

    class TRpsFactory: public ISignalFactory {
    public:
        TRpsFactory(const TString& suffix)
            : Suffix_(suffix)
        {
        }

        TSignal Create(const TString& name) const override {
            return std::make_unique<TSignalRps>(name, Suffix_);
        }

    private:
        TString Suffix_;
    };

    class TSubstringFactory: public ISignalFactory {
    public:
        TSubstringFactory(const TSignalSubstring::TSettings& settings)
            : Settings_(settings)
        {
        }

        TSignal Create(const TString& name) const override {
            return std::make_unique<TSignalSubstring>(name, Settings_);
        }

    private:
        const TSignalSubstring::TSettings Settings_;
    };

    class TSumFactory: public ISignalFactory {
    public:
        TSumFactory(const TString& suffix)
            : Suffix_(suffix)
        {
        }

        TSignal Create(const TString& name) const override {
            return std::make_unique<TSignalSum>(name, Suffix_);
        }

    private:
        TString Suffix_;
    };
}
