#include "mem_storage.h"

#include <passport/infra/libs/cpp/unistat/builder.h>

namespace NPassport::NXunistater {
    static bool ShouldUpdateTime(TMemStorage::ESignalType type, const TMemStorage::TSettings& settings) {
        switch (type) {
            case TMemStorage::ESignalType::Absolute:
                return settings.AbsTtl != TDuration();
            case TMemStorage::ESignalType::Diff:
                return settings.DiffTtl != TDuration();
        }
    }

    TMemStorage::TMemStorage(const TMemStorage::TSettings& settings)
        : Settings_(settings)
    {
    }

    void TMemStorage::AddValue(const TStringBuf key, double value, TInstant now) {
        auto updateTime = [this, now](TCounter& c) {
            if (ShouldUpdateTime(c.Type, Settings_)) {
                c.LastUpdate.store(now, std::memory_order_relaxed);
            }
        };

        {
            std::shared_lock lock(Mutex_);
            auto it = Storage_.find(key);
            if (it != Storage_.end()) {
                UpdateValue(*it->second, value);
                updateTime(*it->second);
                return;
            }
        }

        std::unique_lock lock(Mutex_);
        auto [it, inserted] = Storage_.emplace(key, nullptr);
        if (inserted) {
            const ESignalType type = GetSignalType(key);
            it->second = std::make_unique<TCounter>(type, value);
        } else {
            UpdateValue(*it->second, value);
        }
        updateTime(*it->second);
    }

    void TMemStorage::AddUnistat(NUnistat::TBuilder& builder, TInstant now) {
        TUnistatHelper unistat(Settings_, now);

        {
            std::shared_lock lock(Mutex_);
            unistat.AddUnistat(builder, Storage_);
        }

        if (unistat.NeedCleanup()) {
            std::unique_lock lock(Mutex_);
            unistat.Cleanup(Storage_);
        }
    }

    void TMemStorage::IncErrors() {
        ++StoringErrors_;
    }

    ui64 TMemStorage::GetErrors() const {
        return StoringErrors_.GetValue();
    }

    TMemStorage::ESignalType TMemStorage::GetSignalType(const TStringBuf key) {
        const TStringBuf suffix = TStringBuf(key).RNextTok('_');
        Y_ENSURE(suffix.size() < key.size(),
                 "key is malformed: suffix is bad: '" << suffix << "' in '" << key << "'");

        if (suffix.size() == 4) {
            if (suffix.StartsWith('d')) {
                return ESignalType::Diff;
            }
            if (suffix.StartsWith('a')) {
                return ESignalType::Absolute;
            }
        }

        if (suffix == "summ" || suffix == "hgram") {
            return ESignalType::Diff;
        }
        if (suffix == "max") {
            return ESignalType::Absolute;
        }

        ythrow yexception() << "key is malformed: suffix is bad: '" << suffix << "' in '" << key << "'";
    }

    void TMemStorage::UpdateValue(TCounter& counter, double value) {
        switch (counter.Type) {
            case ESignalType::Absolute:
                counter.Value.store(value, std::memory_order_relaxed);
                break;
            case ESignalType::Diff:
                counter.Value.AddValue(value);
                break;
        }
    }

    TUnistatHelper::TUnistatHelper(const TMemStorage::TSettings& settings, TInstant now)
        : IsExpirableAbs_(ShouldUpdateTime(TMemStorage::ESignalType::Absolute, settings))
        , IsExpirableDiff_(ShouldUpdateTime(TMemStorage::ESignalType::Diff, settings))
        , ExpAbs_(now - settings.AbsTtl)
        , ExpDiff_(now - settings.DiffTtl)
    {
    }

    void TUnistatHelper::AddUnistat(NUnistat::TBuilder& builder, const TMemStorage::TStorage& storage) {
        for (const auto& pair : storage) {
            switch (pair.second->Type) {
                case TMemStorage::ESignalType::Absolute:
                    if (IsExpirableAbs_ && pair.second->IsExpired(ExpAbs_)) {
                        ToDelete_.push_back(pair.first);
                        continue;
                    }
                    break;
                case TMemStorage::ESignalType::Diff:
                    if (IsExpirableDiff_ && pair.second->IsExpired(ExpDiff_)) {
                        ToDelete_.push_back(pair.first);
                        continue;
                    }
                    break;
            }

            builder.AddRow(pair.first, pair.second->Value.GetValue());
        }
    }

    void TUnistatHelper::Cleanup(TMemStorage::TStorage& storage) const {
        for (const TString& s : ToDelete_) {
            auto it = storage.find(s);
            if (it == storage.end()) {
                continue;
            }

            switch (it->second->Type) {
                case TMemStorage::ESignalType::Absolute:
                    if (!it->second->IsExpired(ExpAbs_)) {
                        continue;
                    }
                    break;
                case TMemStorage::ESignalType::Diff:
                    if (!it->second->IsExpired(ExpDiff_)) {
                        continue;
                    }
                    break;
            }

            storage.erase(it);
        }
    }

    bool TUnistatHelper::NeedCleanup() const {
        return !ToDelete_.empty();
    }
}
