#pragma once

#include <passport/infra/libs/cpp/utils/atomic.h>

#include <library/cpp/containers/stack_vector/stack_vec.h>

#include <util/datetime/base.h>
#include <util/generic/hash.h>
#include <util/generic/strbuf.h>
#include <util/generic/string.h>

#include <shared_mutex>

namespace NPassport::NUnistat {
    class TBuilder;
}

namespace NPassport::NXunistater {
    class TMemStorage {
    public:
        struct TSettings {
            TDuration AbsTtl;
            TDuration DiffTtl;
        };

        TMemStorage(const TSettings& settings);
        virtual ~TMemStorage() = default;

        virtual void AddValue(const TStringBuf key, double value, TInstant now = TInstant::Now());

        void AddUnistat(NUnistat::TBuilder& builder, TInstant now = TInstant::Now());

        void IncErrors();
        ui64 GetErrors() const;

        enum class ESignalType {
            Absolute,
            Diff,
        };

        struct TCounter {
            TCounter(ESignalType type, double value)
                : Type(type)
                , Value(value)
            {
            }

            bool IsExpired(TInstant exp) const {
                return LastUpdate.load(std::memory_order_relaxed) < exp;
            }

            const ESignalType Type;
            NUtils::TAtomicNum<double> Value;
            std::atomic<TInstant> LastUpdate = {};
        };

        static ESignalType GetSignalType(const TStringBuf key);
        static void UpdateValue(TCounter& counter, double value);

        using TStorage = THashMap<TString, std::unique_ptr<TCounter>>;

    private:
        NUtils::TAtomicNum<> StoringErrors_;

        mutable std::shared_mutex Mutex_;
        TStorage Storage_;

        const TSettings Settings_;
    };

    struct TUnistatHelper {
        TUnistatHelper(const TMemStorage::TSettings& settings, TInstant now);

        void AddUnistat(NUnistat::TBuilder& builder, const TMemStorage::TStorage& storage);
        void Cleanup(TMemStorage::TStorage& storage) const;
        bool NeedCleanup() const;

    private:
        const bool IsExpirableAbs_;
        const bool IsExpirableDiff_;

        const TInstant ExpAbs_;
        const TInstant ExpDiff_;

        TStackVec<TString> ToDelete_;
    };
}
