#include "xunistater.h"

#include "config/list.h"
#include "config/parsers.h"
#include "config/validator.h"
#include "storage/request_parser.h"
#include "storage/serializer.h"

#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/xml/config.h>

#include <library/cpp/http/misc/httpcodes.h>

namespace NPassport::NXunistater {
    TXunistater::TXunistater() = default;
    TXunistater::~TXunistater() = default;

    void TXunistater::Init(const NXml::TConfig& config) {
        const TString configPath = R"(/config/component)";

        InitLogger(config, configPath);

        try {
            NCfg::ValidateConfig(config);

            InitMiscOptions(config, configPath);
            InitParsers(config, configPath);
            InitStorages(config, configPath);

            InitFailed_ = false;
            TLog::Info("Successfully started");
        } catch (const std::exception& e) {
            TLog::Error("init() failed: %s", e.what());
        }
    }

    void TXunistater::HandleRequest(NCommon::TRequest& req) {
        if (InitFailed_.load(std::memory_order_relaxed)) {
            req.SetStatus(HTTP_INTERNAL_SERVER_ERROR);
            req.Write("Failed to init");
            return;
        }

        ChooseMethod(req);
    }

    void TXunistater::AddUnistat(NUnistat::TBuilder& builder) {
        auto add = [&builder](const auto& st, const TStringBuf name) {
            if (!st.empty()) {
                ui64 errors = 0;
                for (const auto& pair : st) {
                    errors += pair.second.Signals->GetErrors();
                }
                builder.AddRow(name, errors);
            }
        };

        add(Parsers_, "errors.parsing_dmmm");
        add(Storages_, "errors.storing_dmmm");
    }

    void TXunistater::ChooseMethod(NCommon::TRequest& req) {
        const TStringBuf method = req.GetRequestMethod();

        if (method == "GET") {
            HandleMethodGet(req);
            return;
        }

        if (method == "POST") {
            HandleMethodPost(req);
            return;
        }

        req.SetStatus(HTTP_NOT_FOUND);
        req.Write(TStringBuilder() << "unknown method: " << method);
        TLog::Warning() << "unknown method: " << method;
    }

    void TXunistater::HandleMethodGet(NCommon::TRequest& req) const {
        TStringBuf path = req.GetPath();
        if (path.size() > 1) {
            path.ChopSuffix("/");
        }

        if (path == "/ping") {
            req.SetStatus(HTTP_OK);
            return;
        }

        const TString res = HandleSignals(path);
        if (res) {
            req.SetStatus(HTTP_OK);
            req.Write(res);
            return;
        }

        req.SetStatus(HTTP_NOT_FOUND);
        req.Write(TStringBuilder() << "unknown path in GET: " << path);
    }

    void TXunistater::HandleMethodPost(NCommon::TRequest& req) {
        TStringBuf path = req.GetPath();
        if (path.size() > 1) {
            path.ChopSuffix("/");
        }

        auto it = Storages_.find(path);
        if (it != Storages_.end()) {
            const auto resp = HandleMemStorage(*it->second.Signals, req.GetRequestBody());
            req.SetStatus(resp.second ? HTTP_OK : HTTP_BAD_REQUEST);
            req.Write(resp.first);
            return;
        }

        req.SetStatus(HTTP_NOT_FOUND);
        req.Write(TStringBuilder() << "unknown path in POST: " << path);
        TLog::Warning() << "unknown path in POST: " << path;
    }

    TString TXunistater::HandleSignals(const TStringBuf path) const {
        auto add = [](const TStringBuf path, const auto& map, NUnistat::TBuilder& builder) -> std::pair<bool, std::optional<ui64>> {
            const auto range = map.equal_range(path);
            std::optional<ui64> errors;

            for (auto it = range.first; it != range.second; ++it) {
                it->second.Signals->AddUnistat(builder);

                if (it->second.ShowErrors == EShowErrorSignal::True) {
                    ui64 val = it->second.Signals->GetErrors();
                    if (errors) {
                        *errors += val;
                    } else {
                        errors = val;
                    }
                }
            }

            return {range.first != range.second, errors};
        };

        TString res;
        {
            NUnistat::TBuilder builder(res);
            auto [parserFound, parserErrors] = add(path, Parsers_, builder);
            auto [storageFound, storageErrors] = add(path, Storages_, builder);

            if (!parserFound && !storageFound) {
                return TString();
            }

            if (parserErrors) {
                builder.AddRow("__errors.parsing_dmmm", *parserErrors);
            }
            if (storageErrors) {
                builder.AddRow("__errors.storing_dmmm", *storageErrors);
            }
        }

        return res;
    }

    std::pair<TString, bool> TXunistater::HandleMemStorage(TMemStorage& storage, const TStringBuf body) {
        try {
            TRequestParser::Parse(storage, body);
        } catch (const std::exception& e) {
            storage.IncErrors();
            TLog::Warning() << "mem_storage exception: " << e.what();
            return {TSerializer::Error(e.what()), false};
        }

        return {TSerializer::Ok(), true};
    }

    void TXunistater::InitLogger(const NXml::TConfig& config, const TString& xpath) {
        config.InitCommonLog(xpath + "/logger");
    }

    void TXunistater::InitMiscOptions(const NXml::TConfig& config, const TString& xpath) {
        AreHttpPathesUnique_ = config.AsBool(xpath + "/unique_http_path", true);
    }

    static TString GetHttpPath(const NXml::TConfig& config, const TString& xpath) {
        TString res = config.AsString(xpath + "/http_path");
        if (res.back() == '/') {
            res.pop_back();
        }
        return res;
    }

    void TXunistater::InitParsers(const NXml::TConfig& config, const TString& xpath) {
        auto add = [this](const NXml::TConfig& config, const TString& xpath, std::unique_ptr<TParserBase> p) {
            const TString httpPath = GetHttpPath(config, xpath);
            CheckUniqueHttpPath(httpPath);

            TSignals<TParserBase> res{
                std::move(p),
                ShowErrorSignal(config, xpath),
            };
            Parsers_.emplace(httpPath, std::move(res));
        };

        for (const NCfgList::TTsvParser& parser : NCfgList::ListTsvParsers(config, xpath)) {
            add(config, parser.Xpath, InitTsvParser(config, parser));
        }
        for (const NCfgList::TTskvParser& parser : NCfgList::ListTskvParsers(config, xpath)) {
            add(config, parser.Xpath, InitTskvParser(config, parser));
        }
    }

    void TXunistater::InitStorages(const NXml::TConfig& config, const TString& xpath) {
        for (const TString& stor : NCfg::GetPrettyXpathSubkeys(config, xpath + "/mem_storage")) {
            const TString httpPath = GetHttpPath(config, stor);
            CheckUniqueHttpPath(httpPath);

            TMemStorage::TSettings settings;
            settings.AbsTtl = TDuration::Seconds(config.AsInt(stor + "/ttl/abs", 0));
            settings.DiffTtl = TDuration::Seconds(config.AsInt(stor + "/ttl/diff", 0));

            TSignals<TMemStorage> res{
                std::make_unique<TMemStorage>(settings),
                ShowErrorSignal(config, xpath),
            };

            Y_ENSURE(Storages_.emplace(httpPath, std::move(res)).second,
                     "http handle is duplicated for mem_storage: " << httpPath);
        }
    }

    std::unique_ptr<TParserBase> TXunistater::InitTsvParser(const NXml::TConfig& config,
                                                            const NCfgList::TTsvParser& xpath) {
        const int period = config.AsInt(xpath.Xpath + "/period_to_check_file_ms", 500);

        return NTsv::TParser::CreateUnique(
            {
                TDuration::MilliSeconds(period),
                "xunist_parser",
            },
            NCfg::GetBaseParserSettings(config, xpath.Xpath),
            NCfg::GetTsvParserArgs(config, xpath));
    }

    std::unique_ptr<TParserBase> TXunistater::InitTskvParser(const NXml::TConfig& config,
                                                             const NCfgList::TTskvParser& xpath) {
        const int period = config.AsInt(xpath.Xpath + "/period_to_check_file_ms", 500);

        return NTskv::TParser::CreateUnique(
            {
                TDuration::MilliSeconds(period),
                "xunist_tparser",
            },
            NCfg::GetBaseParserSettings(config, xpath.Xpath),
            NCfg::GetTskvParserArgs(config, xpath));
    }

    void TXunistater::CheckUniqueHttpPath(const TString& httpPath) const {
        if (AreHttpPathesUnique_) {
            Y_ENSURE(!Parsers_.contains(httpPath) && !Storages_.contains(httpPath),
                     "http path is duplicated: " << httpPath);
        }
    }

    EShowErrorSignal TXunistater::ShowErrorSignal(const NXml::TConfig& config, const TString& xpath) {
        return config.AsBool(xpath + "/show_error_signal", true)
                   ? EShowErrorSignal::True
                   : EShowErrorSignal::False;
    }
}
