#include <passport/infra/daemons/xunistater/src/parsers/tskv.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/system/fs.h>

using namespace NPassport;
using namespace NPassport::NXunistater;
using namespace NPassport::NXunistater::NTskv::NConds;

Y_UNIT_TEST_SUITE(Conditions) {
    Y_UNIT_TEST(trim) {
        UNIT_ASSERT_VALUES_EQUAL(
            "field1!=value1 and field2==value2 and ( ( ( field===kek ) ) or field17=kek ) and field18 or !field17",
            TrimExpression(" \n   field1  !  =  value1  \n and   field2  ==     value2  and((  (field === kek   )    )  or field17  =  kek  )  and  field18  or  ! field17 "));
    }

    Y_UNIT_TEST(validate) {
        UNIT_ASSERT_EXCEPTION_CONTAINS(ValidateEqualCount("field="),
                                       yexception,
                                       "Expression can't ends with '='");
        UNIT_ASSERT_EXCEPTION_CONTAINS(ValidateEqualCount("field=="),
                                       yexception,
                                       "Expression can't ends with '=='");
        UNIT_ASSERT_EXCEPTION_CONTAINS(ValidateEqualCount("field=value"),
                                       yexception,
                                       "Expression can contain equals only like this '=='");
        UNIT_ASSERT_EXCEPTION_CONTAINS(ValidateEqualCount("field===value"),
                                       yexception,
                                       "Expression can contain equals only like this '=='");
        UNIT_ASSERT_NO_EXCEPTION(ValidateEqualCount("field==value"));

        UNIT_ASSERT_EXCEPTION_CONTAINS(ValidateEqualCount("!=value"),
                                       yexception,
                                       "Expression can't starts with '!='");
        UNIT_ASSERT_EXCEPTION_CONTAINS(ValidateEqualCount("field!="),
                                       yexception,
                                       "Expression can't ends with '!='");
        UNIT_ASSERT_EXCEPTION_CONTAINS(ValidateEqualCount("field!=!value"),
                                       yexception,
                                       "Expression can contain not-equals only like this '!='");
        UNIT_ASSERT_NO_EXCEPTION(ValidateEqualCount("field!=value"));

        UNIT_ASSERT_EXCEPTION_CONTAINS(ValidateBraceCount(")"),
                                       yexception,
                                       "Opening and closing braces mismatched");
        UNIT_ASSERT_EXCEPTION_CONTAINS(ValidateBraceCount("(  ))"),
                                       yexception,
                                       "Opening and closing braces mismatched");
        UNIT_ASSERT_EXCEPTION_CONTAINS(ValidateBraceCount("((  )"),
                                       yexception,
                                       "Opening and closing braces mismatched");
        UNIT_ASSERT_EXCEPTION_CONTAINS(ValidateBraceCount(")("),
                                       yexception,
                                       "Opening and closing braces mismatched");
        UNIT_ASSERT_EXCEPTION_CONTAINS(ValidateBraceCount("("),
                                       yexception,
                                       "Opening and closing braces mismatched");
        UNIT_ASSERT_NO_EXCEPTION(ValidateBraceCount("()((()))"));
    }

    Y_UNIT_TEST(Condition_kv_eq) {
        TKeyValue vals;
        TCondition cond = TKvEquals::Create("key1", "val1");
        UNIT_ASSERT_VALUES_EQUAL("key1==val1", cond->Serialize());
        UNIT_ASSERT(!cond->Match(vals));

        vals["key2"] = "val2";
        UNIT_ASSERT(!cond->Match(vals));
        vals["key1"] = "val100500";
        UNIT_ASSERT(!cond->Match(vals));
        vals["key1"] = "val1";
        UNIT_ASSERT(cond->Match(vals));
    }

    Y_UNIT_TEST(Condition_kv_noteq) {
        TKeyValue vals;
        TCondition cond = TKvNotEquals::Create("key1", "val1");
        UNIT_ASSERT_VALUES_EQUAL("key1!=val1", cond->Serialize());
        UNIT_ASSERT(cond->Match(vals));

        vals["key2"] = "val2";
        UNIT_ASSERT(cond->Match(vals));
        vals["key1"] = "val100500";
        UNIT_ASSERT(cond->Match(vals));
        vals["key1"] = "val1";
        UNIT_ASSERT(!cond->Match(vals));
    }

    Y_UNIT_TEST(Condition_or) {
        TCondition cond = TOr::Create(
            TKvEquals::Create("key1", "val1"),
            TKvEquals::Create("key3", "val3"));
        UNIT_ASSERT_VALUES_EQUAL("(key1==val1 || key3==val3)", cond->Serialize());

        TKeyValue vals;
        UNIT_ASSERT(!cond->Match(vals));
        vals["key1"] = "val1";
        UNIT_ASSERT(cond->Match(vals));
        vals["key3"] = "val3";
        UNIT_ASSERT(cond->Match(vals));
        vals["key1"] = "kek";
        UNIT_ASSERT(cond->Match(vals));
        vals["key3"] = "kek2";
        UNIT_ASSERT(!cond->Match(vals));
    }

    Y_UNIT_TEST(Condition_and) {
        TCondition cond = TAnd::Create(
            TKvEquals::Create("key1", "val1"),
            TKvEquals::Create("key3", "val3"));
        UNIT_ASSERT_VALUES_EQUAL("(key1==val1 && key3==val3)", cond->Serialize());

        TKeyValue vals;
        UNIT_ASSERT(!cond->Match(vals));
        vals["key1"] = "val1";
        UNIT_ASSERT(!cond->Match(vals));
        vals["key3"] = "val3";
        UNIT_ASSERT(cond->Match(vals));
        vals["key1"] = "kek";
        UNIT_ASSERT(!cond->Match(vals));
        vals["key3"] = "kek2";
        UNIT_ASSERT(!cond->Match(vals));
    }

    Y_UNIT_TEST(Condition_true) {
        TCondition cond = TTrue::Create();
        UNIT_ASSERT_VALUES_EQUAL("true", cond->Serialize());

        TKeyValue vals;
        UNIT_ASSERT(cond->Match(vals));
        vals["key1"] = "val1";
        UNIT_ASSERT(cond->Match(vals));
        vals["key3"] = "val3";
        UNIT_ASSERT(cond->Match(vals));
        vals["key1"] = "kek";
        UNIT_ASSERT(cond->Match(vals));
        vals["key3"] = "kek2";
        UNIT_ASSERT(cond->Match(vals));
    }

    Y_UNIT_TEST(Condition_contain) {
        TCondition cond = TContainKey::Create("key1");
        UNIT_ASSERT_VALUES_EQUAL("CONTAIN(key1)", cond->Serialize());

        TKeyValue vals;
        UNIT_ASSERT(!cond->Match(vals));
        vals["key2"] = "val2";
        UNIT_ASSERT(!cond->Match(vals));
        vals["key1"] = "val1";
        UNIT_ASSERT(cond->Match(vals));
        vals["key1"] = "kek";
        UNIT_ASSERT(cond->Match(vals));
        vals.erase("key2");
        UNIT_ASSERT(cond->Match(vals));
        vals.erase("key1");
        UNIT_ASSERT(!cond->Match(vals));
    }

    Y_UNIT_TEST(Condition_not_contain) {
        TCondition cond = TNotContainKey::Create("key1");
        UNIT_ASSERT_VALUES_EQUAL("NOT_CONTAIN(key1)", cond->Serialize());

        TKeyValue vals;
        UNIT_ASSERT(cond->Match(vals));
        vals["key2"] = "val2";
        UNIT_ASSERT(cond->Match(vals));
        vals["key1"] = "val1";
        UNIT_ASSERT(!cond->Match(vals));
        vals["key1"] = "kek";
        UNIT_ASSERT(!cond->Match(vals));
        vals.erase("key2");
        UNIT_ASSERT(!cond->Match(vals));
        vals.erase("key1");
        UNIT_ASSERT(cond->Match(vals));
    }

    Y_UNIT_TEST(empty) {
        TCondition cond = ExpressionToCondition("");
        UNIT_ASSERT_VALUES_EQUAL("true", cond->Serialize());

        TKeyValue vals;
        UNIT_ASSERT(cond->Match(vals));

        vals["kek"] = "foo";
        UNIT_ASSERT(cond->Match(vals));
    }

    Y_UNIT_TEST(one) {
        TCondition cond = ExpressionToCondition("field1==value1");
        UNIT_ASSERT_VALUES_EQUAL("field1==value1", cond->Serialize());

        TKeyValue vals;
        UNIT_ASSERT(!cond->Match(vals));

        vals["kek"] = "foo";
        UNIT_ASSERT(!cond->Match(vals));

        vals["field1"] = "foo";
        UNIT_ASSERT(!cond->Match(vals));

        vals["field1"] = "value1";
        UNIT_ASSERT(cond->Match(vals));
    }

    Y_UNIT_TEST(one_not) {
        TCondition cond = ExpressionToCondition("field1!=value1");
        UNIT_ASSERT_VALUES_EQUAL("field1!=value1", cond->Serialize());

        TKeyValue vals;
        UNIT_ASSERT(cond->Match(vals));

        vals["kek"] = "foo";
        UNIT_ASSERT(cond->Match(vals));

        vals["field1"] = "foo";
        UNIT_ASSERT(cond->Match(vals));

        vals["field1"] = "value1";
        UNIT_ASSERT(!cond->Match(vals));
    }

    Y_UNIT_TEST(two_and) {
        TCondition cond = ExpressionToCondition("( field1==value1 and field2==value2 )");
        UNIT_ASSERT_VALUES_EQUAL("(field1==value1 && field2==value2)", cond->Serialize());

        TKeyValue vals;
        UNIT_ASSERT(!cond->Match(vals));

        vals["kek"] = "foo";
        UNIT_ASSERT(!cond->Match(vals));

        vals["field1"] = "foo";
        UNIT_ASSERT(!cond->Match(vals));

        vals["field1"] = "value1";
        UNIT_ASSERT(!cond->Match(vals));

        vals["field2"] = "kek";
        UNIT_ASSERT(!cond->Match(vals));

        vals["field2"] = "value2";
        UNIT_ASSERT(cond->Match(vals));
    }

    Y_UNIT_TEST(two_and_not) {
        TCondition cond = ExpressionToCondition("( field1==value1 and field2!=value2 )");
        UNIT_ASSERT_VALUES_EQUAL("(field1==value1 && field2!=value2)", cond->Serialize());

        TKeyValue vals;
        UNIT_ASSERT(!cond->Match(vals));

        vals["kek"] = "foo";
        UNIT_ASSERT(!cond->Match(vals));

        vals["field1"] = "foo";
        UNIT_ASSERT(!cond->Match(vals));

        vals["field1"] = "value1";
        UNIT_ASSERT(cond->Match(vals));

        vals["field2"] = "kek";
        UNIT_ASSERT(cond->Match(vals));

        vals["field2"] = "value2";
        UNIT_ASSERT(!cond->Match(vals));
    }

    Y_UNIT_TEST(three_and) {
        TCondition cond = ExpressionToCondition("( field1==value1 and ( field2==value2 and field3==value3 ) )");
        UNIT_ASSERT_VALUES_EQUAL("(field1==value1 && (field2==value2 && field3==value3))", cond->Serialize());

        TKeyValue vals;
        UNIT_ASSERT(!cond->Match(vals));

        vals["kek"] = "foo";
        UNIT_ASSERT(!cond->Match(vals));

        vals["field1"] = "foo";
        UNIT_ASSERT(!cond->Match(vals));

        vals["field1"] = "value1";
        UNIT_ASSERT(!cond->Match(vals));

        vals["field2"] = "kek";
        UNIT_ASSERT(!cond->Match(vals));

        vals["field2"] = "value2";
        UNIT_ASSERT(!cond->Match(vals));

        vals["field3"] = "value3";
        UNIT_ASSERT(cond->Match(vals));

        cond = ExpressionToCondition("( field1==value1 and field2==value2 ) and field3==value3");
        UNIT_ASSERT_VALUES_EQUAL("((field1==value1 && field2==value2) && field3==value3)", cond->Serialize());
        UNIT_ASSERT(cond->Match(vals));

        cond = ExpressionToCondition("field1==value1 and field2==value2 and field3==value3");
        UNIT_ASSERT_VALUES_EQUAL("((field1==value1 && field2==value2) && field3==value3)", cond->Serialize());
        UNIT_ASSERT(cond->Match(vals));
    }

    Y_UNIT_TEST(three_and_or) {
        TCondition cond1 = ExpressionToCondition("( field1==value1 or field2==value2 and field3==value3 )");
        UNIT_ASSERT_VALUES_EQUAL("(field1==value1 || (field2==value2 && field3==value3))", cond1->Serialize());

        TCondition cond2 = ExpressionToCondition("field1==value1 and field2==value2 or field3==value3");
        UNIT_ASSERT_VALUES_EQUAL("((field1==value1 && field2==value2) || field3==value3)", cond2->Serialize());

        TKeyValue vals;
        UNIT_ASSERT(!cond1->Match(vals));
        UNIT_ASSERT(!cond2->Match(vals));

        vals["kek"] = "foo";
        UNIT_ASSERT(!cond1->Match(vals));
        UNIT_ASSERT(!cond2->Match(vals));

        vals["field1"] = "foo";
        UNIT_ASSERT(!cond1->Match(vals));
        UNIT_ASSERT(!cond2->Match(vals));

        vals["field1"] = "value1";
        UNIT_ASSERT(cond1->Match(vals));
        UNIT_ASSERT(!cond2->Match(vals));

        vals["field2"] = "kek";
        UNIT_ASSERT(cond1->Match(vals));
        UNIT_ASSERT(!cond2->Match(vals));

        vals["field2"] = "value2";
        UNIT_ASSERT(cond1->Match(vals));
        UNIT_ASSERT(cond2->Match(vals));

        vals["field1"] = "foo";
        UNIT_ASSERT(!cond1->Match(vals));
        UNIT_ASSERT(!cond2->Match(vals));

        vals["field3"] = "value3";
        UNIT_ASSERT(cond1->Match(vals));
        UNIT_ASSERT(cond2->Match(vals));
    }

    Y_UNIT_TEST(complicated) {
        UNIT_ASSERT_STRINGS_EQUAL(
            "(field1==value1 || (field2==value2 && field3==value3))",
            TranslateExpressionForTest("( ( field1==value1 ) or ( field2==value2 ) and ( field3==value3 ) )"));

        UNIT_ASSERT_STRINGS_EQUAL(
            "((field1==value1 && field2==kek) || (field2==value2 && field3==value3))",
            TranslateExpressionForTest("( ( field1==value1 and field2==kek ) or field2==value2 and field3==value3 )"));
        UNIT_ASSERT_STRINGS_EQUAL(
            "((field1==value1 && field2==kek) || (field2==value2 && field3==value3))",
            TranslateExpressionForTest("( field1==value1 and field2==kek or (field2==value2 and field3==value3) )"));
        UNIT_ASSERT_STRINGS_EQUAL(
            "((field1==value1 && field2==kek) || (field2==value2 && field3==value3))",
            TranslateExpressionForTest("field1==value1 and field2==kek or field2==value2 and field3==value3"));

        UNIT_ASSERT_STRINGS_EQUAL(
            "(mode==restore_semi_auto && ((action==finished_with_state && (state==complete_pdd || (state==password_change_forbidden || (state==complete_social || state==complete_autoregistered)))) || (action==finished_with_error && (error==account.without_password || (error==account.disabled || error==account.not_found)))))",
            TranslateExpressionForTest("mode==restore_semi_auto and ( (action==finished_with_state and (state==complete_pdd or state==password_change_forbidden or state==complete_social or state==complete_autoregistered)) or (action==finished_with_error and (error==account.without_password or error==account.disabled or error == account.not_found)))"));

        UNIT_ASSERT_STRINGS_EQUAL(
            "((mode==restore_semi_auto && CONTAIN(mode)) && ((action==finished_with_state && (state==complete_pdd || (state==password_change_forbidden || (state==complete_social || state==complete_autoregistered)))) || (action==finished_with_error && (error==account.without_password || (error==account.disabled || error==account.not_found)))))",
            TranslateExpressionForTest("mode==restore_semi_auto and mode and ( (action==finished_with_state and (state==complete_pdd or state==password_change_forbidden or state==complete_social or state==complete_autoregistered)) or (action==finished_with_error and (error==account.without_password or error==account.disabled or error == account.not_found)))"));
        UNIT_ASSERT_STRINGS_EQUAL(
            "(((mode==restore_semi_auto && CONTAIN(mode)) && NOT_CONTAIN(status)) && ((action==finished_with_state && (state==complete_pdd || (state==password_change_forbidden || (state==complete_social || state==complete_autoregistered)))) || (action==finished_with_error && (error==account.without_password || (error==account.disabled || error==account.not_found)))))",
            TranslateExpressionForTest("mode==restore_semi_auto and mode and !status and ( (action==finished_with_state and (state==complete_pdd or state==password_change_forbidden or state==complete_social or state==complete_autoregistered)) or (action==finished_with_error and (error==account.without_password or error==account.disabled or error == account.not_found)))"));
    }

    Y_UNIT_TEST(errors) {
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            ExpressionToCondition("and field1==value1 or field2==value2 (and field3==value3"),
            yexception,
            "Illegal expression. Allowed: opened brace OR key-value expression OR bool function\n"
            "'and field1==value1 or field2==value2 (and field3==value3'\n"
            " ^");
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            ExpressionToCondition("( field1==value1 or field2==value2 (and field3==value3 )"),
            yexception,
            "Illegal expression. Allowed: close brace OR and/or\n"
            "'( field1==value1 or field2==value2 (and field3==value3 )'\n"
            "                                    ^");
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            ExpressionToCondition("( field1==value1 or field2==value2 ) field3==value3"),
            yexception,
            "Illegal expression. Allowed: close brace OR and/or\n"
            "'( field1==value1 or field2==value2 ) field3==value3'\n"
            "                                     ^");
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            ExpressionToCondition("field1==value1 or ) field2==value2 and field3==value3"),
            yexception,
            "Illegal expression. Allowed: opened brace OR key-value expression OR bool function\n"
            "'field1==value1 or ) field2==value2 and field3==value3'\n"
            "                   ^");
    }
}
