#include <passport/infra/daemons/xunistater/src/parsers/parser_base.h>
#include <passport/infra/daemons/xunistater/src/signal_providers/regex.h>
#include <passport/infra/daemons/xunistater/src/signal_providers/rps.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/system/fs.h>

using namespace NPassport;
using namespace NPassport::NXunistater;

Y_UNIT_TEST_SUITE(ParserBase) {
    const TString filename = "ololo.log";

    class TTestParser: public TParserBase {
    public:
        TTestParser(TSignal signal)
            : TParserBase({filename, 255})
            , Signal(std::move(signal))
        {
        }

        void AddUnistat(NUnistat::TBuilder& builder) const override {
            Signal->AddUnistat(builder);
        }

        bool ProcessLine(TStringBuf line) override {
            Signal->Process(line);
            return true;
        }

        void Flush() override {
            Signal->FlushBuffer();
        }

        using TParserBase::Run;

        TSignal Signal;
    };

    Y_UNIT_TEST(nofile) {
        NFs::Remove(filename);
        UNIT_ASSERT_EXCEPTION_CONTAINS(TTestParser({}),
                                       yexception,
                                       R"(can't open "ololo.log" with mode RdOnly|Seq)");
    }

    Y_UNIT_TEST(run) {
        NFs::Remove(filename);
        TFileOutput file(filename);
        for (int i = 0; i < 500; ++i) {
            file << "---" << Endl;
        }

        TTestParser parser(std::make_unique<TSignalRps>("rps", "dhhh"));
        ISignalProvider* sig = parser.Signal.get();
        UNIT_ASSERT_STRINGS_EQUAL(R"([["rps_dhhh",0]])", sig->SerializeForTest());

        parser.Run();
        UNIT_ASSERT_STRINGS_EQUAL(R"([["rps_dhhh",0]])", sig->SerializeForTest());

        for (int i = 0; i < 1000; ++i) {
            file << "---" << Endl;
        }
        UNIT_ASSERT_STRINGS_EQUAL(R"([["rps_dhhh",0]])", sig->SerializeForTest());

        parser.Run();
        UNIT_ASSERT_STRINGS_EQUAL(R"([["rps_dhhh",1000]])", sig->SerializeForTest());
    }

    Y_UNIT_TEST(flush) {
        class TTestSignal: public TSignalRps {
        public:
            TTestSignal()
                : TSignalRps("kek", "dmmm")
            {
            }

            TErrorMsg Process(TStringBuf field) override {
                ++Lines;
                return TSignalRps::Process(field);
            }

            void FlushBuffer() noexcept override {
                ++Flushes;
                TSignalRps::FlushBuffer();
            }

            ui64 Lines = 0;
            ui64 Flushes = 0;
        };

        NFs::Remove(filename);
        TFileOutput file(filename);

        TTestParser parser(std::make_unique<TTestSignal>());
        TTestSignal* tstSig = static_cast<TTestSignal*>(parser.Signal.get());

        for (int i = 0; i < 3; ++i) {
            file << "---" << Endl;
        }
        UNIT_ASSERT_NO_EXCEPTION(parser.Run());

        UNIT_ASSERT_VALUES_EQUAL(3, tstSig->Lines);
        UNIT_ASSERT_VALUES_EQUAL(1, tstSig->Flushes);

        for (int i = 0; i < 1000; ++i) {
            file << "---" << Endl;
        }
        UNIT_ASSERT_NO_EXCEPTION(parser.Run());

        UNIT_ASSERT_VALUES_EQUAL(1003, tstSig->Lines);
        UNIT_ASSERT_VALUES_EQUAL(5, tstSig->Flushes);
    }

    Y_UNIT_TEST(buffered) {
        NFs::Remove(filename);
        TFileOutput file(filename);
        for (int i = 0; i < 100; ++i) {
            file << "---" << Endl;
        }

        TTestParser parser(std::make_unique<TSignalRps>("kek", "dhhh"));
        ISignalProvider* sig = parser.Signal.get();

        file << "-- -" << Endl;
        parser.Run();
        UNIT_ASSERT_STRINGS_EQUAL(R"([["kek_dhhh",1]])", sig->SerializeForTest());

        file << "-- " << Flush;
        parser.Run();
        UNIT_ASSERT_STRINGS_EQUAL(R"([["kek_dhhh",1]])", sig->SerializeForTest());

        file << "-" << Flush;
        parser.Run();
        UNIT_ASSERT_STRINGS_EQUAL(R"([["kek_dhhh",1]])", sig->SerializeForTest());

        file << Endl;
        parser.Run();
        UNIT_ASSERT_STRINGS_EQUAL(R"([["kek_dhhh",2]])", sig->SerializeForTest());

        file << Endl << Flush;
        parser.Run();
        UNIT_ASSERT_STRINGS_EQUAL(R"([["kek_dhhh",3]])", sig->SerializeForTest());
    }
}
