#include <passport/infra/daemons/xunistater/src/parsers/tskv.h>
#include <passport/infra/daemons/xunistater/src/signal_providers/rps.h>
#include <passport/infra/daemons/xunistater/src/signal_providers/string_set.h>

#include <passport/infra/libs/cpp/xml/config.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/system/fs.h>

using namespace NPassport;
using namespace NPassport::NXunistater;
using namespace NPassport::NXunistater::NTskv;

Y_UNIT_TEST_SUITE(Tskv) {
    const TString filename = "ololo.log";
    ISignalProvider* GetSignal(TConditionSet::TSignals& signals, TString key) {
        return signals.find(key)->second.get();
    }

    class TTestParser: public TTskvParser {
    public:
        TTestParser(TConditions&& procs)
            : TTskvParser({filename, 255}, {std::move(procs)})
        {
        }

        using TTskvParser::ParseLine;
        using TTskvParser::ProccessCondition;
        using TTskvParser::ProcessLine;
        using TTskvParser::Run;
    };

    Y_UNIT_TEST(proccessCondition) {
        NFs::Remove(filename);
        TFileOutput file(filename);

        TKeyValue vals;
        TConditionSet cond;
        cond.Condition = NConds::TTrue::Create();
        cond.Signals.emplace("field1", std::make_unique<TSignalRps>("name1", "dhhh"));
        cond.Signals.emplace("field2", std::make_unique<TSignalRps>("name2", "dhhh"));

        TTestParser parser({});
        parser.ProccessCondition(vals, cond, "kek");
        GetSignal(cond.Signals, "field1")->FlushBuffer();
        GetSignal(cond.Signals, "field2")->FlushBuffer();
        UNIT_ASSERT_VALUES_EQUAL(R"([["name1_dhhh",0]])", GetSignal(cond.Signals, "field1")->SerializeForTest());
        UNIT_ASSERT_VALUES_EQUAL(R"([["name2_dhhh",0]])", GetSignal(cond.Signals, "field2")->SerializeForTest());

        vals["field1"] = "";
        parser.ProccessCondition(vals, cond, "kek");
        GetSignal(cond.Signals, "field1")->FlushBuffer();
        GetSignal(cond.Signals, "field2")->FlushBuffer();
        UNIT_ASSERT_VALUES_EQUAL(R"([["name1_dhhh",1]])", GetSignal(cond.Signals, "field1")->SerializeForTest());
        UNIT_ASSERT_VALUES_EQUAL(R"([["name2_dhhh",0]])", GetSignal(cond.Signals, "field2")->SerializeForTest());

        cond.Condition = NConds::TKvEquals::Create("field2", "val2");
        parser.ProccessCondition(vals, cond, "kek");
        GetSignal(cond.Signals, "field1")->FlushBuffer();
        GetSignal(cond.Signals, "field2")->FlushBuffer();
        UNIT_ASSERT_VALUES_EQUAL(R"([["name1_dhhh",1]])", GetSignal(cond.Signals, "field1")->SerializeForTest());
        UNIT_ASSERT_VALUES_EQUAL(R"([["name2_dhhh",0]])", GetSignal(cond.Signals, "field2")->SerializeForTest());

        vals["field2"] = "val2";
        parser.ProccessCondition(vals, cond, "kek");
        GetSignal(cond.Signals, "field1")->FlushBuffer();
        GetSignal(cond.Signals, "field2")->FlushBuffer();
        UNIT_ASSERT_VALUES_EQUAL(R"([["name1_dhhh",2]])", GetSignal(cond.Signals, "field1")->SerializeForTest());
        UNIT_ASSERT_VALUES_EQUAL(R"([["name2_dhhh",1]])", GetSignal(cond.Signals, "field2")->SerializeForTest());
    }

    Y_UNIT_TEST(proccess_errs) {
        class TExceptSignal: public ISignalProvider {
        public:
            TErrorMsg Process(TStringBuf) override {
                ythrow yexception() << "kekkek";
            }
            void FlushBuffer() noexcept override {
            }
            void AddUnistat(NUnistat::TBuilder&) const override {
            }
        };
        class TErrSignal: public ISignalProvider {
        public:
            TErrorMsg Process(TStringBuf) override {
                return "kekkek";
            }
            void FlushBuffer() noexcept override {
            }
            void AddUnistat(NUnistat::TBuilder&) const override {
            }
        };

        NFs::Remove(filename);
        TFileOutput file(filename);

        TKeyValue vals = {
            {"field1", ""},
            {"field2", "val2"},
            {"field3", "val3"},
        };
        TConditionSet cond;
        cond.Condition = NConds::TTrue::Create();
        cond.Signals.emplace("field1", std::make_unique<TExceptSignal>());
        cond.Signals.emplace("field2", std::make_unique<TErrSignal>());
        cond.Signals.emplace("field3", std::make_unique<TSignalRps>("name3", "dhhh"));

        TTestParser parser({});
        parser.ProccessCondition(vals, cond, "kek");
        GetSignal(cond.Signals, "field3")->FlushBuffer();
        UNIT_ASSERT_VALUES_EQUAL(R"([["name3_dhhh",1]])", GetSignal(cond.Signals, "field3")->SerializeForTest());
    }

    Y_UNIT_TEST(processLine) {
        NFs::Remove(filename);
        TFileOutput file(filename);

        TConditionSet cond;
        cond.Condition = NConds::TTrue::Create();
        cond.Signals.emplace("field1", std::make_unique<TSignalRps>("name1", "dhhh"));
        ISignalProvider* sig = GetSignal(cond.Signals, "field1");

        TConditions c;
        c.push_back(std::move(cond));
        TTestParser parser(std::move(c));
        UNIT_ASSERT(!parser.ProcessLine("kek"));
        sig->FlushBuffer();
        UNIT_ASSERT_VALUES_EQUAL(R"([["name1_dhhh",0]])", sig->SerializeForTest());

        UNIT_ASSERT(parser.ProcessLine("tskv\t"
                                       "field1=kek"));
        sig->FlushBuffer();
        UNIT_ASSERT_VALUES_EQUAL(R"([["name1_dhhh",1]])", sig->SerializeForTest());

        UNIT_ASSERT(!parser.ProcessLine("tskv\t"
                                        "field1=kek\t"));
        sig->FlushBuffer();
        UNIT_ASSERT_VALUES_EQUAL(R"([["name1_dhhh",1]])", sig->SerializeForTest());

        UNIT_ASSERT(!parser.ProcessLine("tskv\t"));
        sig->FlushBuffer();
        UNIT_ASSERT_VALUES_EQUAL(R"([["name1_dhhh",1]])", sig->SerializeForTest());

        UNIT_ASSERT(parser.ProcessLine("field1=kek"));
        sig->FlushBuffer();
        UNIT_ASSERT_VALUES_EQUAL(R"([["name1_dhhh",2]])", sig->SerializeForTest());
    }

    Y_UNIT_TEST(parseLine) {
        TString buf;
        TKeyValue map;

        UNIT_ASSERT_VALUES_EQUAL("",
                                 TTestParser::ParseLine("field1=kek", map, buf));
        UNIT_ASSERT_VALUES_EQUAL("kek", map["field1"]);
        UNIT_ASSERT_VALUES_EQUAL(1, map.size());
        map.clear();

        UNIT_ASSERT_VALUES_EQUAL("Incorrect tskv format",
                                 TTestParser::ParseLine("field1\t=\tkek", map, buf));
        UNIT_ASSERT(map.empty());
    }
}
