#include <passport/infra/daemons/xunistater/src/parsers/tsv.h>
#include <passport/infra/daemons/xunistater/src/signal_providers/regex.h>
#include <passport/infra/daemons/xunistater/src/signal_providers/rps.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/system/fs.h>

using namespace NPassport;
using namespace NPassport::NXunistater;
using namespace NPassport::NXunistater::NTsv;

Y_UNIT_TEST_SUITE(Tsv) {
    const TString filename = "ololo.log";

    class TTestParser: public TTsvParser {
    public:
        TTestParser(TColumnProcessors&& procs)
            : TTsvParser({filename, 255}, {std::move(procs), TSplittingPolicy::Create(' '), {}})
        {
        }

        using TTsvParser::Run;
    };

    Y_UNIT_TEST(except) {
        class TTestSignal: public ISignalProvider {
        public:
            TTestSignal() = default;

            TErrorMsg Process(TStringBuf) override {
                ++Lines;
                ythrow yexception() << "kekkek";
            }
            void FlushBuffer() noexcept override {
                ++Flushes;
            }
            void AddUnistat(NUnistat::TBuilder&) const override {
            }

            ui64 Lines = 0;
            ui64 Flushes = 0;
        };

        NFs::Remove(filename);
        TFileOutput file(filename);

        TColumnProcessors procs;
        procs.push_back({TColumnGetter({}), std::make_unique<TTestSignal>()});
        TTestSignal* tstSig = static_cast<TTestSignal*>(procs.back().Signal.get());
        procs.push_back({TColumnGetter({}), std::make_unique<TSignalRps>("rps", "dhhh")});
        ISignalProvider* rpsSig = procs.back().Signal.get();

        TTestParser parser(std::move(procs));
        for (int i = 0; i < 257; ++i) {
            file << "---" << Endl;
        }
        UNIT_ASSERT_NO_EXCEPTION(parser.Run());

        UNIT_ASSERT_VALUES_EQUAL(257, tstSig->Lines);
        UNIT_ASSERT_VALUES_EQUAL(2, tstSig->Flushes);
        UNIT_ASSERT_STRINGS_EQUAL(R"([["rps_dhhh",257]])", rpsSig->SerializeForTest());
    }

    Y_UNIT_TEST(failProcess) {
        class TTestSignal: public ISignalProvider {
        public:
            TTestSignal() = default;

            TErrorMsg Process(TStringBuf) override {
                ++Lines;
                return {"lol"};
            }
            void FlushBuffer() noexcept override {
                ++Flushes;
            }
            void AddUnistat(NUnistat::TBuilder&) const override {
            }

            ui64 Lines = 0;
            ui64 Flushes = 0;
        };

        NFs::Remove(filename);
        TFileOutput file(filename);

        TColumnProcessors procs;
        procs.push_back({TColumnGetter({}), std::make_unique<TTestSignal>()});
        TTestSignal* tstSig = static_cast<TTestSignal*>(procs.back().Signal.get());
        procs.push_back({TColumnGetter({}), std::make_unique<TSignalRps>("rps", "_dhhh")});
        ISignalProvider* rpsSig = procs.back().Signal.get();

        TTestParser parser(std::move(procs));
        for (int i = 0; i < 257; ++i) {
            file << "---" << Endl;
        }
        UNIT_ASSERT_NO_EXCEPTION(parser.Run());

        UNIT_ASSERT_VALUES_EQUAL(257, tstSig->Lines);
        UNIT_ASSERT_VALUES_EQUAL(2, tstSig->Flushes);
        UNIT_ASSERT_STRINGS_EQUAL(R"([["rps_dhhh",257]])", rpsSig->SerializeForTest());
    }

    Y_UNIT_TEST(failGet) {
        NFs::Remove(filename);
        TFileOutput file(filename);

        TColumnProcessors procs;
        procs.push_back({TColumnGetter(3), std::make_unique<TSignalRps>("kek", "_dhhh")});
        ISignalProvider* tstSig = procs.back().Signal.get();
        procs.push_back({TColumnGetter({}), std::make_unique<TSignalRps>("rps", "_dhhh")});
        ISignalProvider* rpsSig = procs.back().Signal.get();

        TTestParser parser(std::move(procs));
        for (int i = 0; i < 257; ++i) {
            file << "---" << Endl;
        }
        UNIT_ASSERT_NO_EXCEPTION(parser.Run());

        UNIT_ASSERT_STRINGS_EQUAL(R"([["kek_dhhh",0]])", tstSig->SerializeForTest());
        UNIT_ASSERT_STRINGS_EQUAL(R"([["rps_dhhh",257]])", rpsSig->SerializeForTest());
    }

    Y_UNIT_TEST(split) {
        TSplittingPolicy policy('\t');
        TString str = "\tsome\tcool string";
        TLine line;

        UNIT_ASSERT(policy.ParseLine(str, line));
        UNIT_ASSERT_VALUES_EQUAL(TLine({"", "some", "cool string"}), line);
    }

    Y_UNIT_TEST(regex) {
        TRegexingPolicy policy(R"(\[[^ ]+ [-+]\d{4}\] [^ ]+ (\d+?\.\d{3}) (\d+?\.\d{3}) \d+ [^ ]+ (\/.*?)([\?\&].*)? [^ ]+ \"(\d{3})\" \d+ [^ ]+ \".*?\" [^ ]+ \"(.*?)\")");
        TString str = R"([2019-02-01 +0300] kek 100.500 100.501 17 lol /ping?key=value foo "499" 700 bar "some mega string" hu "some other string")";
        TLine line;

        UNIT_ASSERT(policy.ParseLine(str, line));
        UNIT_ASSERT_VALUES_EQUAL(TLine({"100.500", "100.501", "/ping", "?key=value", "499", "some other string"}),
                                 line);
    }

    Y_UNIT_TEST(ColumnGetter_missing) {
        TColumnGetter g({});

        UNIT_ASSERT_VALUES_EQUAL("full line", g.AsString());
        TLine line;
        std::pair<TStringBuf, TErrorMsg> res;

        res = g.GetField(line, "kek");
        UNIT_ASSERT_VALUES_EQUAL(res.first, "kek");
        UNIT_ASSERT_VALUES_EQUAL(res.second, "");

        line.push_back("ololo");
        res = g.GetField(line, "kek3");
        UNIT_ASSERT_VALUES_EQUAL(res.first, "kek3");
        UNIT_ASSERT_VALUES_EQUAL(res.second, "");
    }

    Y_UNIT_TEST(ColumnGetter_specified) {
        UNIT_ASSERT_EXCEPTION_CONTAINS(TColumnGetter(0), yexception, "columnNo_ must be > 0: 0");

        TColumnGetter g(2);

        UNIT_ASSERT_VALUES_EQUAL("field=2", g.AsString());
        TLine line;
        std::pair<TStringBuf, TErrorMsg> res;

        res = g.GetField(line, "kek");
        UNIT_ASSERT_VALUES_EQUAL(res.first, "");
        UNIT_ASSERT_VALUES_EQUAL(res.second, "line has not required field=2");

        line.push_back("ololo");
        res = g.GetField(line, "kek3");
        UNIT_ASSERT_VALUES_EQUAL(res.first, "");
        UNIT_ASSERT_VALUES_EQUAL(res.second, "line has not required field=2");

        line.push_back("kekeke");
        res = g.GetField(line, "kek3");
        UNIT_ASSERT_VALUES_EQUAL(res.first, "kekeke");
        UNIT_ASSERT_VALUES_EQUAL(res.second, "");
    }
}
