#include <passport/infra/daemons/xunistater/ut/auto_uninstater.h>

#include <passport/infra/daemons/xunistater/src/signal_providers/string_set.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NXunistater;

Y_UNIT_TEST_SUITE(StringSet) {
    Y_UNIT_TEST(line) {
        TSignalStringSet signal("_dhhh", {TSignalStringSet::TSettings::EUnquote::True, {}, {}, {}, "ololo.", ""});
        TAutoUnistater au(signal);

        UNIT_ASSERT_STRINGS_EQUAL(R"([])", signal.SerializeForTest());

        UNIT_ASSERT_STRINGS_EQUAL("", signal.Process("kek"));

        for (int i = 0; i < 256; ++i) {
            UNIT_ASSERT_STRINGS_EQUAL("", signal.Process("qwerty"));
        }
        UNIT_ASSERT_STRINGS_EQUAL(R"([])", signal.SerializeForTest());

        signal.FlushBuffer();
        UNIT_ASSERT_STRINGS_EQUAL(R"([["ololo.kek_dhhh",1],["ololo.qwerty_dhhh",256]])", signal.SerializeForTest());

        for (int i = 0; i < 256; ++i) {
            UNIT_ASSERT_STRINGS_EQUAL("", signal.Process("qwe"));
            UNIT_ASSERT_STRINGS_EQUAL("", signal.Process("qwerty"));
        }

        signal.FlushBuffer();
        UNIT_ASSERT_STRINGS_EQUAL(R"([["ololo.kek_dhhh",1],["ololo.qwe_dhhh",256],["ololo.qwerty_dhhh",512]])", signal.SerializeForTest());

        signal.FlushBuffer();
        UNIT_ASSERT_STRINGS_EQUAL(R"([["ololo.kek_dhhh",1],["ololo.qwe_dhhh",256],["ololo.qwerty_dhhh",512]])", signal.SerializeForTest());
    }

    Y_UNIT_TEST(unquote) {
        TSignalStringSet signal("_dhhh", {TSignalStringSet::TSettings::EUnquote::True, {}, {}, {}, "ololo.", ""});
        TAutoUnistater au(signal);

        UNIT_ASSERT_STRINGS_EQUAL(R"([])", signal.SerializeForTest());

        UNIT_ASSERT_STRINGS_EQUAL("", signal.Process("qwe"));
        UNIT_ASSERT_STRINGS_EQUAL("", signal.Process(R"("qwerty")"));

        signal.FlushBuffer();
        UNIT_ASSERT_STRINGS_EQUAL(R"([["ololo.qwe_dhhh",1],["ololo.qwerty_dhhh",1]])", signal.SerializeForTest());
    }

    Y_UNIT_TEST(dont_unquote) {
        TSignalStringSet signal("_dhhh", {TSignalStringSet::TSettings::EUnquote::False, {}, {}, {}, "ololo.", ""});
        TAutoUnistater au(signal);

        UNIT_ASSERT_STRINGS_EQUAL(R"([])", signal.SerializeForTest());

        UNIT_ASSERT_STRINGS_EQUAL("", signal.Process("qwe"));
        UNIT_ASSERT_STRINGS_EQUAL("", signal.Process(R"("qwerty")"));

        signal.FlushBuffer();
        UNIT_ASSERT_STRINGS_EQUAL(R"([["ololo.\"qwerty\"_dhhh",1],["ololo.qwe_dhhh",1]])", signal.SerializeForTest());
    }

    Y_UNIT_TEST(regex_filter_matched) {
        TSignalStringSet signal("_dhhh",
                                {
                                    TSignalStringSet::TSettings::EUnquote::False,
                                    R"(\w{4,})",
                                    TSignalStringSet::TSettings::ERegexSelect::Matched,
                                    {},
                                    "ololo.",
                                    "",
                                });
        TAutoUnistater au(signal);

        UNIT_ASSERT_STRINGS_EQUAL(R"([])", signal.SerializeForTest());

        UNIT_ASSERT_STRINGS_EQUAL("", signal.Process("qwe"));
        UNIT_ASSERT_STRINGS_EQUAL("", signal.Process(R"("qwerty")"));

        signal.FlushBuffer();
        UNIT_ASSERT_STRINGS_EQUAL(R"([["ololo.\"qwerty\"_dhhh",1]])", signal.SerializeForTest());
    }

    Y_UNIT_TEST(regex_filter_nonmatched) {
        TSignalStringSet signal("_dhhh",
                                {
                                    TSignalStringSet::TSettings::EUnquote::False,
                                    R"(\w{4,})",
                                    TSignalStringSet::TSettings::ERegexSelect::NonMatched,
                                    {},
                                    "ololo.",
                                    "",
                                });
        TAutoUnistater au(signal);

        UNIT_ASSERT_STRINGS_EQUAL(R"([])", signal.SerializeForTest());

        UNIT_ASSERT_STRINGS_EQUAL("", signal.Process("qwe"));
        UNIT_ASSERT_STRINGS_EQUAL("", signal.Process(R"("qwerty")"));

        signal.FlushBuffer();
        UNIT_ASSERT_STRINGS_EQUAL(R"([["ololo.qwe_dhhh",1]])", signal.SerializeForTest());
    }

    Y_UNIT_TEST(regex_filter_matched_unquote) {
        TSignalStringSet signal("_dhhh",
                                {
                                    TSignalStringSet::TSettings::EUnquote::True,
                                    R"("q.*")",
                                    TSignalStringSet::TSettings::ERegexSelect::Matched,
                                    {},
                                    "ololo.",
                                    "",
                                });
        TAutoUnistater au(signal);

        UNIT_ASSERT_STRINGS_EQUAL(R"([])", signal.SerializeForTest());

        UNIT_ASSERT_STRINGS_EQUAL("", signal.Process("qwe"));
        UNIT_ASSERT_STRINGS_EQUAL("", signal.Process(R"("qwerty")"));

        signal.FlushBuffer();
        UNIT_ASSERT_STRINGS_EQUAL(R"([])", signal.SerializeForTest());
    }

    Y_UNIT_TEST(regex_filter_matched_dontunquote) {
        TSignalStringSet signal("_dmmm",
                                {
                                    TSignalStringSet::TSettings::EUnquote::False,
                                    R"("q.*")",
                                    TSignalStringSet::TSettings::ERegexSelect::Matched,
                                    {},
                                    "ololo.",
                                    "",
                                });
        TAutoUnistater au(signal);

        UNIT_ASSERT_STRINGS_EQUAL(R"([])", signal.SerializeForTest());

        UNIT_ASSERT_STRINGS_EQUAL("", signal.Process("qwe"));
        UNIT_ASSERT_STRINGS_EQUAL("", signal.Process(R"("qwerty")"));

        signal.FlushBuffer();
        UNIT_ASSERT_STRINGS_EQUAL(R"([["ololo.\"qwerty\"_dmmm",1]])", signal.SerializeForTest());
    }

    Y_UNIT_TEST(persist) {
        TSignalStringSet signal("_dhhh",
                                {
                                    TSignalStringSet::TSettings::EUnquote::False,
                                    {},
                                    {},
                                    {"qwe", "kek"},
                                    "ololo.",
                                    "",
                                });
        TAutoUnistater au(signal);

        UNIT_ASSERT_STRINGS_EQUAL(R"([["ololo.kek_dhhh",0],["ololo.qwe_dhhh",0]])", signal.SerializeForTest());

        UNIT_ASSERT_STRINGS_EQUAL("", signal.Process("qwe"));
        UNIT_ASSERT_STRINGS_EQUAL("", signal.Process(R"(qwerty)"));

        signal.FlushBuffer();
        UNIT_ASSERT_STRINGS_EQUAL(R"([["ololo.kek_dhhh",0],["ololo.qwe_dhhh",1],["ololo.qwerty_dhhh",1]])", signal.SerializeForTest());
    }

    Y_UNIT_TEST(usePattern) {
        TSignalStringSet signal("_dhhh",
                                {
                                    TSignalStringSet::TSettings::EUnquote::False,
                                    R"(/(\w+)\?consumer=([^&]+)&uid=1)",
                                    TSignalStringSet::TSettings::ERegexSelect::Matched,
                                    {},
                                    "",
                                    "some.$2.mega_$1 signal",
                                });
        TAutoUnistater au(signal);

        UNIT_ASSERT_STRINGS_EQUAL(R"([])", signal.SerializeForTest());

        UNIT_ASSERT_STRINGS_EQUAL("", signal.Process("qwe"));
        UNIT_ASSERT_STRINGS_EQUAL("", signal.Process(R"("/ping?consumer=kokoko&uid=1")"));

        signal.FlushBuffer();
        UNIT_ASSERT_STRINGS_EQUAL(R"([["some.kokoko.mega_ping signal_dhhh",1]])", signal.SerializeForTest());
    }

    Y_UNIT_TEST(usePatternWithGroupAtTheBegining) {
        TSignalStringSet signal("_dhhh",
                                {
                                    TSignalStringSet::TSettings::EUnquote::False,
                                    R"(/(\w+)\?consumer=([^&]+)&uid=1)",
                                    TSignalStringSet::TSettings::ERegexSelect::Matched,
                                    {},
                                    "",
                                    "$2.mega_$1",
                                });
        TAutoUnistater au(signal);

        UNIT_ASSERT_STRINGS_EQUAL(R"([])", signal.SerializeForTest());

        UNIT_ASSERT_STRINGS_EQUAL("", signal.Process("qwe"));
        UNIT_ASSERT_STRINGS_EQUAL("", signal.Process(R"("/ping?consumer=kokoko&uid=1")"));

        signal.FlushBuffer();
        UNIT_ASSERT_STRINGS_EQUAL(R"([["kokoko.mega_ping_dhhh",1]])", signal.SerializeForTest());
    }
}
