#include <passport/infra/daemons/xunistater/src/storage/mem_storage.h>

#include <passport/infra/libs/cpp/unistat/builder.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NXunistater;

Y_UNIT_TEST_SUITE(Storage) {
    Y_UNIT_TEST(type) {
        UNIT_ASSERT_VALUES_EQUAL(TMemStorage::ESignalType::Diff, TMemStorage::GetSignalType("kek_dmmm"));
        UNIT_ASSERT_VALUES_EQUAL(TMemStorage::ESignalType::Diff, TMemStorage::GetSignalType("kek_dhhh"));
        UNIT_ASSERT_VALUES_EQUAL(TMemStorage::ESignalType::Diff, TMemStorage::GetSignalType("kek_summ"));
        UNIT_ASSERT_VALUES_EQUAL(TMemStorage::ESignalType::Diff, TMemStorage::GetSignalType("kek_hgram"));

        UNIT_ASSERT_VALUES_EQUAL(TMemStorage::ESignalType::Absolute, TMemStorage::GetSignalType("kek_avvx"));
        UNIT_ASSERT_VALUES_EQUAL(TMemStorage::ESignalType::Absolute, TMemStorage::GetSignalType("kek_axxx"));
        UNIT_ASSERT_VALUES_EQUAL(TMemStorage::ESignalType::Absolute, TMemStorage::GetSignalType("kek_max"));

        UNIT_ASSERT_EXCEPTION_CONTAINS(TMemStorage::GetSignalType("kek_mmmm"),
                                       yexception,
                                       "key is malformed: suffix is bad: 'mmmm' in 'kek_mmmm'");
        UNIT_ASSERT_EXCEPTION_CONTAINS(TMemStorage::GetSignalType(""),
                                       yexception,
                                       "key is malformed: suffix is bad: '' in ''");
        UNIT_ASSERT_EXCEPTION_CONTAINS(TMemStorage::GetSignalType("kekdmmm"),
                                       yexception,
                                       "key is malformed: suffix is bad: 'kekdmmm' in 'kekdmmm'");
        UNIT_ASSERT_EXCEPTION_CONTAINS(TMemStorage::GetSignalType("dmmm"),
                                       yexception,
                                       "key is malformed: suffix is bad: 'dmmm' in 'dmmm'");
    }

    Y_UNIT_TEST(updateAbs) {
        TMemStorage::TCounter c(TMemStorage::ESignalType::Absolute, 0);
        for (size_t idx = 0; idx < 10; ++idx) {
            TMemStorage::UpdateValue(c, 10.5);
        }
        UNIT_ASSERT_VALUES_EQUAL(10.5, c.Value.GetValue());
    }

    Y_UNIT_TEST(updateDiff) {
        TMemStorage::TCounter c(TMemStorage::ESignalType::Diff, 0);
        for (size_t idx = 0; idx < 10; ++idx) {
            TMemStorage::UpdateValue(c, 10.5);
        }
        UNIT_ASSERT_VALUES_EQUAL(105, c.Value.GetValue());
    }

    static TString Serialize(TMemStorage& stor, TInstant now = TInstant::Now()) {
        TString res;
        NUnistat::TBuilder stats(res);
        stor.AddUnistat(stats, now);
        return res;
    }

    Y_UNIT_TEST(common) {
        TMemStorage stor({});

        UNIT_ASSERT_VALUES_EQUAL(R"([])", Serialize(stor));

        stor.AddValue("key1_dhhh", 100);
        stor.AddValue("key2_avvx", 1000000000000);
        stor.AddValue("key1_dhhh", 0.25);
        stor.AddValue("key2_avvx", 2);
        UNIT_ASSERT_VALUES_EQUAL(R"([["key2_avvx",2.0],["key1_dhhh",100.25]])", Serialize(stor));
    }

    Y_UNIT_TEST(commonWithTtlAbs) {
        TMemStorage::TSettings s;
        s.AbsTtl = TDuration::Minutes(1);

        TMemStorage stor(s);
        const TInstant now = TInstant::Now();

        UNIT_ASSERT_VALUES_EQUAL(R"([])", Serialize(stor, now));

        stor.AddValue("key1_dhhh", 100, now);
        stor.AddValue("key2_avvx", 1000000000000, now);
        stor.AddValue("key1_dhhh", 0.25, now);
        stor.AddValue("key2_avvx", 2, now);
        UNIT_ASSERT_VALUES_EQUAL(R"([["key2_avvx",2.0],["key1_dhhh",100.25]])", Serialize(stor, now));
        UNIT_ASSERT_VALUES_EQUAL(R"([["key1_dhhh",100.25]])", Serialize(stor, now + TDuration::Minutes(2)));
        UNIT_ASSERT_VALUES_EQUAL(R"([["key1_dhhh",100.25]])", Serialize(stor, now));
    }

    Y_UNIT_TEST(commonWithTtlDiff) {
        TMemStorage::TSettings s;
        s.DiffTtl = TDuration::Minutes(3);

        TMemStorage stor(s);
        const TInstant now = TInstant::Now();

        UNIT_ASSERT_VALUES_EQUAL(R"([])", Serialize(stor, now));

        stor.AddValue("key1_dhhh", 100, now);
        stor.AddValue("key2_avvx", 1000000000000, now);
        stor.AddValue("key1_dhhh", 0.25, now);
        stor.AddValue("key2_avvx", 2, now);
        UNIT_ASSERT_VALUES_EQUAL(R"([["key2_avvx",2.0],["key1_dhhh",100.25]])", Serialize(stor, now));
        UNIT_ASSERT_VALUES_EQUAL(R"([["key2_avvx",2.0]])", Serialize(stor, now + TDuration::Minutes(5)));
        UNIT_ASSERT_VALUES_EQUAL(R"([["key2_avvx",2.0]])", Serialize(stor, now));
    }

    Y_UNIT_TEST(commonWithTtlAbsDiff) {
        TMemStorage::TSettings s;
        s.AbsTtl = TDuration::Minutes(1);
        s.DiffTtl = TDuration::Minutes(3);

        TMemStorage stor(s);
        const TInstant now = TInstant::Now();

        UNIT_ASSERT_VALUES_EQUAL(R"([])", Serialize(stor, now));

        stor.AddValue("key1_dhhh", 100, now);
        stor.AddValue("key2_avvx", 1000000000000, now);
        stor.AddValue("key1_dhhh", 0.25, now);
        stor.AddValue("key2_avvx", 2, now);
        UNIT_ASSERT_VALUES_EQUAL(R"([["key2_avvx",2.0],["key1_dhhh",100.25]])", Serialize(stor, now));
        UNIT_ASSERT_VALUES_EQUAL(R"([["key1_dhhh",100.25]])", Serialize(stor, now + TDuration::Minutes(2)));
        UNIT_ASSERT_VALUES_EQUAL(R"([])", Serialize(stor, now + TDuration::Minutes(5)));
        UNIT_ASSERT_VALUES_EQUAL(R"([])", Serialize(stor, now));
    }
}
