package filter

import (
	_ "embed"
	"encoding/json"
	"fmt"
	"os"

	"github.com/santhosh-tekuri/jsonschema/v5"

	"a.yandex-team.ru/library/go/core/xerrors"
)

var JSONSchema *jsonschema.Schema

type logicOperatorContainer struct {
	LogicOperator
}

type compareOperatorContainer struct {
	CompareOperator
}

type parsedFilter struct {
	LogicOp *logicOperatorContainer `json:"logic_op"`
	Args    []FilterContainer       `json:"args"`

	Field     string                    `json:"field"`
	CompareOp *compareOperatorContainer `json:"compare_op"`
	Values    []string                  `json:"values"`
}

func FromJSON(data []byte) (Filter, error) {
	var container FilterContainer
	err := json.Unmarshal(data, &container)
	return container.Filter, err
}

func (op *logicOperatorContainer) UnmarshalJSON(data []byte) error {
	var value string
	if err := json.Unmarshal(data, &value); err != nil {
		return err
	}

	res, ok := logicOpFromString[value]
	if !ok {
		return xerrors.Errorf("unknown logic operator type '%s'", value)
	}

	op.LogicOperator = res

	return nil
}

func (op *compareOperatorContainer) UnmarshalJSON(data []byte) error {
	var value string
	if err := json.Unmarshal(data, &value); err != nil {
		return err
	}

	res, ok := compareOpFromString[value]
	if !ok {
		return xerrors.Errorf("unknown compare operator type '%s'", value)
	}

	op.CompareOperator = res

	return nil
}

func (filter *FilterContainer) UnmarshalJSON(data []byte) error {
	var parsed parsedFilter
	if err := json.Unmarshal(data, &parsed); err != nil {
		return err
	}

	var count int

	if parsed.LogicOp != nil && parsed.Args != nil {
		if len(parsed.Args) == 0 {
			return xerrors.Errorf("logic operator must have at least one argument")
		}

		filter.Filter = &LogicOpFilter{
			LogicOp: parsed.LogicOp.LogicOperator,
			Args:    parsed.Args,
		}

		count++
	}

	if parsed.CompareOp != nil && parsed.Values != nil {
		if parsed.Field == "" {
			return xerrors.Errorf("field name can't be empty")
		}
		if len(parsed.Values) == 0 {
			return xerrors.Errorf("field filter must have at least one value")
		}

		filter.Filter = &FieldFilter{
			Field:     parsed.Field,
			CompareOp: parsed.CompareOp.CompareOperator,
			Values:    parsed.Values,
		}

		count++
	}

	if count == 0 {
		return xerrors.New("json does not match any filter type")
	}
	if count > 1 {
		return xerrors.New("json matches more than one filter type")
	}

	return nil
}

func ValidateJSONSchema(data []byte) error {
	var v interface{}
	if err := json.Unmarshal(data, &v); err != nil {
		return err
	}
	if err := JSONSchema.Validate(v); err != nil {
		return err
	}

	return nil
}

//go:embed filter.schema.json
var rawJSONSchema []byte

func init() {
	s, err := jsonschema.CompileString("schema.json", string(rawJSONSchema))
	if err != nil {
		_, _ = fmt.Fprintf(os.Stderr, "Failed to compile filter json schema: %s", err.Error())
		os.Exit(1)
	}
	JSONSchema = s
}
