package filter

import (
	"strconv"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
)

type validationMeta struct {
	CurrentDepth uint64
}

type Schema struct {
	Fields     FieldsSchema
	CheckDepth bool
	MaxDepth   uint64
}

type FieldsSchema map[string]*FieldSchema
type FieldValidation func(string) error

type FieldSchema struct {
	CompareOperators CompareOperatorsSchema
	Validate         FieldValidation
}

func isNumeric(value string) error {
	if _, err := strconv.Atoi(value); err != nil {
		return xerrors.Errorf("should be numeric")
	}
	return nil
}

type CompareOperatorsSchema map[CompareOperator]interface{}

var NumberFieldSchema = &FieldSchema{
	CompareOperators: CompareOperatorsSchema{
		Equal:    nil,
		NotEqual: nil,
	},
	Validate: isNumeric,
}

var NumberFieldWithCompareSchema = &FieldSchema{
	CompareOperators: CompareOperatorsSchema{
		Equal:    nil,
		NotEqual: nil,
		More:     nil,
		Less:     nil,
	},
	Validate: isNumeric,
}

var StringFieldSchema = &FieldSchema{
	CompareOperators: CompareOperatorsSchema{
		Equal:      nil,
		NotEqual:   nil,
		Contains:   nil,
		StartsWith: nil,
	},
}

func EnumFieldSchema(enumValues map[string]interface{}) *FieldSchema {
	return &FieldSchema{
		CompareOperators: CompareOperatorsSchema{
			Equal:    nil,
			NotEqual: nil,
		},

		Validate: func(value string) error {
			if _, exists := enumValues[value]; !exists {
				return xerrors.Errorf("should be from values list")
			}
			return nil
		},
	}
}

var DatetimeFieldSchema = &FieldSchema{
	CompareOperators: CompareOperatorsSchema{
		Equal:    nil,
		NotEqual: nil,
		More:     nil,
		Less:     nil,
	},

	Validate: func(value string) error {
		if _, err := time.Parse(time.RFC3339, value); err != nil {
			return xerrors.Errorf("should be RFC3339 formatted datetime")
		}
		return nil
	},
}

func (schema *Schema) Validate(filter Filter) error {
	return filter.validateImpl(schema, validationMeta{})
}

func (filter *LogicOpFilter) validateImpl(schema *Schema, meta validationMeta) error {
	if schema.CheckDepth && meta.CurrentDepth >= schema.MaxDepth {
		return xerrors.Errorf("filter depth exceeded limit: %d", schema.MaxDepth)
	}

	meta.CurrentDepth++
	for _, arg := range filter.Args {
		if err := arg.validateImpl(schema, meta); err != nil {
			return err
		}
	}

	return nil
}

func (filter *FieldFilter) validateImpl(schema *Schema, meta validationMeta) error {
	if schema.Fields == nil {
		return nil
	}

	fieldSchema, exists := schema.Fields[filter.Field]
	if !exists {
		return xerrors.Errorf("field '%s' is not permitted", filter.Field)
	}
	if fieldSchema == nil {
		return nil
	}

	if _, exists = fieldSchema.CompareOperators[filter.CompareOp]; !exists {
		return xerrors.Errorf("compare operator '%s' is not permitted for field '%s'", filter.CompareOp.ToString(), filter.Field)
	}

	if len(filter.Values) == 0 {
		return xerrors.Errorf("no values in '%s'", filter.Field)
	}

	if !filter.CompareOp.ValuesArrayAllowed() && len(filter.Values) > 1 {
		return xerrors.Errorf("compare operator '%s' requires single value in '%s'", filter.CompareOp.ToString(), filter.Field)
	}

	if fieldSchema.Validate != nil {
		for _, value := range filter.Values {
			if err := fieldSchema.Validate(value); err != nil {
				return xerrors.Errorf("field '%s' has not allowed value '%s': %w", filter.Field, value, err)
			}
		}
	}

	return nil
}
