package loggers

import (
	"context"
	"fmt"
	"strconv"
	"strings"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/model"
	"a.yandex-team.ru/passport/infra/libs/go/tskv"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type TskvFormat string

const (
	TskvActionFormat TskvFormat = "action"
)

type ActionType string

const (
	ActionCreate ActionType = "create"
	ActionUpdate ActionType = "update"
	ActionDelete ActionType = "delete"
)

type EntityType string

const (
	GateEntity         EntityType = "gate"
	RouteEntity        EntityType = "route"
	FallbackEntity     EntityType = "fallback"
	BlockedPhoneEntity EntityType = "blockedphone"
	RegionEntity       EntityType = "region"
	TemplateEntity     EntityType = "template"
)

type TskvLog struct {
	logger log.Logger
}

type TskvContextLog struct {
	logger *TskvLog
	uid    string
}

func NewTskvLog(config logger.Config) (*TskvLog, error) {
	config.DisablePrintingLevel = true
	config.DisablePrintingTime = true

	res, err := logger.CreateLog(config)
	return &TskvLog{
		logger: res,
	}, err
}

func (log *TskvLog) Write(line *tskv.Line) {
	log.logger.Info(line.String())
}

func (log *TskvLog) WithContext(ctx context.Context) *TskvContextLog {
	uid := "-"
	if ut := tvm.ContextUserTicket(ctx); ut != nil {
		uid = strconv.FormatUint(uint64(ut.DefaultUID), 10)
	}

	return &TskvContextLog{
		logger: log,
		uid:    uid,
	}
}

func (log *TskvContextLog) LogAction(
	t time.Time,
	provider string,
	action ActionType,
	entity EntityType,
	id string,
	fields map[string]string,
) {
	line := tskv.NewTskvLine(string(TskvActionFormat)).
		AddValue("date", t.Local().Format("2006-01-02T15:04:05"), false).
		AddValue("unixtime", strconv.FormatInt(t.Unix(), 10), false).
		AddValue("user", log.uid, false).
		AddValue("provider", provider, true).
		AddValue("action", string(action), false).
		AddValue("entity", string(entity), false).
		AddValue("id", id, true).
		AddValues(fields, true)

	log.logger.Write(line)
}

func (log *TskvContextLog) LogDeleteEntity(t time.Time, provider string, entity EntityType, deleted []model.EntityID) {
	for _, id := range deleted {
		log.LogAction(t, provider, ActionDelete, entity, id, nil)
	}
}

func (log *TskvContextLog) LogGatesAction(
	t time.Time,
	provider string,
	deleted []model.EntityID,
	created []*model.Gate,
	updated []*model.Gate,
) {
	log.LogDeleteEntity(t, provider, GateEntity, deleted)

	for _, gate := range created {
		log.LogAction(t, provider, ActionCreate, GateEntity, gate.ID, makeGateFields(gate))
	}

	for _, gate := range updated {
		log.LogAction(t, provider, ActionUpdate, GateEntity, gate.ID, makeGateFields(gate))
	}
}

func makeGateFields(gate *model.Gate) map[string]string {
	extra := make([]string, 0, len(gate.Extra))
	for key, val := range gate.Extra {
		extra = append(extra, fmt.Sprint(key, "=", val))
	}

	return map[string]string{
		"alias":      gate.Alias,
		"alpha_name": gate.AlphaName,
		"consumer":   gate.Consumer,
		"contractor": gate.Contractor,
		"extra":      strings.Join(extra, ";"),
	}
}

func (log *TskvContextLog) LogRoutesAction(
	t time.Time,
	provider string,
	deleted []model.EntityID,
	created []*model.Route,
	updated []*model.Route,
) {
	log.LogDeleteEntity(t, provider, RouteEntity, deleted)

	for _, route := range created {
		log.LogAction(t, provider, ActionCreate, RouteEntity, route.ID, makeRouteFields(route))
	}

	for _, route := range updated {
		log.LogAction(t, provider, ActionUpdate, RouteEntity, route.ID, makeRouteFields(route))
	}
}

func makeRouteFields(route *model.Route) map[string]string {
	return map[string]string{
		"destination": route.PhonePrefix,
		"gates":       strings.Join(route.Gates, ","),
		"weight":      strconv.FormatInt(int64(route.Weight), 10),
		"mode":        route.Mode,
	}
}

func (log *TskvContextLog) LogFallbacksAction(
	t time.Time,
	provider string,
	deleted []model.EntityID,
	created []*model.Fallback,
	updated []*model.Fallback,
) {
	log.LogDeleteEntity(t, provider, FallbackEntity, deleted)

	for _, fallback := range created {
		log.LogAction(t, provider, ActionCreate, FallbackEntity, fallback.ID, makeFallbackFields(fallback))
	}

	for _, fallback := range updated {
		log.LogAction(t, provider, ActionUpdate, FallbackEntity, fallback.ID, makeFallbackFields(fallback))
	}
}

func makeFallbackFields(fallback *model.Fallback) map[string]string {
	return map[string]string{
		"srcgate": fallback.SrcGate,
		"srcname": fallback.SrcName,
		"dstgate": fallback.DstGate,
		"dstname": fallback.DstName,
		"order":   strconv.FormatInt(int64(fallback.Order), 10),
	}
}

func (log *TskvContextLog) LogBlockedPhonesAction(
	t time.Time,
	provider string,
	deleted []model.EntityID,
	created []*model.BlockedPhone,
	updated []*model.BlockedPhone,
) {
	log.LogDeleteEntity(t, provider, BlockedPhoneEntity, deleted)

	for _, blockedPhone := range created {
		log.LogAction(t, provider, ActionCreate, BlockedPhoneEntity, blockedPhone.ID, makeBlockedPhoneFields(blockedPhone))
	}

	for _, blockedPhone := range updated {
		log.LogAction(t, provider, ActionUpdate, BlockedPhoneEntity, blockedPhone.ID, makeBlockedPhoneFields(blockedPhone))
	}
}

func makeBlockedPhoneFields(blockedPhone *model.BlockedPhone) map[string]string {
	return map[string]string{
		"phone":       blockedPhone.PhoneNumber,
		"block_type":  string(blockedPhone.BlockType),
		"block_until": strconv.FormatInt(blockedPhone.BlockUntil.Unix(), 10),
	}
}

func (log *TskvContextLog) LogRegionsAction(
	t time.Time,
	provider string,
	deleted []model.EntityID,
	created []*model.Region,
	updated []*model.Region,
) {

	log.LogDeleteEntity(t, provider, RegionEntity, deleted)

	for _, region := range created {
		log.LogAction(t, provider, ActionCreate, RegionEntity, region.ID, makeRegionFields(region))
	}

	for _, region := range updated {
		log.LogAction(t, provider, ActionUpdate, RegionEntity, region.ID, makeRegionFields(region))
	}
}

func makeRegionFields(region *model.Region) map[string]string {
	return map[string]string{
		"prefix": region.Prefix,
		"name":   region.Name,
	}
}

func makeTemplateFields(template *model.Template) map[string]string {
	return map[string]string{
		"text":               template.Text,
		"abc_service":        template.AbcService,
		"sender_meta":        template.SenderMeta,
		"fields_description": template.FieldsDescription,
	}
}

func makeTemplateUpdateFields(template *model.Template) map[string]string {
	return map[string]string{
		"abc_service":        template.AbcService,
		"sender_meta":        template.SenderMeta,
		"fields_description": template.FieldsDescription,
	}
}

func (log *TskvContextLog) LogTemplatesAction(
	t time.Time,
	provider string,
	created []*model.Template,
	updated []*model.Template,
) {
	for _, template := range created {
		log.LogAction(t, provider, ActionCreate, TemplateEntity, template.ID, makeTemplateFields(template))
	}

	for _, template := range updated {
		log.LogAction(t, provider, ActionUpdate, TemplateEntity, template.ID, makeTemplateUpdateFields(template))
	}
}
