package mysql

import (
	"context"
	"fmt"
	"strconv"

	"github.com/jmoiron/sqlx"

	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/model"
)

const selectAuditBulkQueryTemplate = `
SELECT audit_bulk.bulk_id as bulk_id, author, issue, comment, timestamp, type, entity_id, payload, row_id
FROM audit_bulk JOIN audit_row ON audit_row.bulk_id=audit_bulk.bulk_id WHERE audit_bulk.bulk_id=? ORDER BY row_id
`

const selectAuditChangesInfoQueryTemplate = `
SELECT audit_row.bulk_id as bulk_id, author, issue, comment, timestamp, type, entity_id, row_id
FROM audit_bulk JOIN audit_row ON audit_row.bulk_id=audit_bulk.bulk_id WHERE row_id IN (?) ORDER BY row_id
`

func (provider *Provider) GetAuditBulkInfo(ctx context.Context, id model.EntityID) (*model.AuditBulkInfo, error) {
	// many rows with different changes for one bulk
	var dbResults []GetAuditBulkInfoResultWithPayload

	query, args, err := sqlx.In(selectAuditBulkQueryTemplate, id)
	if err != nil {
		return nil, fmt.Errorf("failed to prepare select audit bulk info query: %s", err)
	}

	_, err = provider.withDriver(func(db *sqlx.DB) (interface{}, error) {
		return nil, db.SelectContext(ctx, &dbResults, db.Rebind(query), args...)
	})
	if err != nil {
		return nil, fmt.Errorf("failed to select audit bulk: %s %s %s", err, query, args)
	}

	if len(dbResults) == 0 {
		return nil, nil
	}

	var bulkInfo model.AuditBulkInfo
	bulkInfo.ID = strconv.FormatInt(dbResults[0].ID.Int64, 10)
	bulkInfo.Author = dbResults[0].Author.String
	bulkInfo.Comment = dbResults[0].Comment.String
	bulkInfo.Issue = dbResults[0].Issue.String
	bulkInfo.Timestamp = dbResults[0].TS.Int64

	bulkInfo.Changes = make(map[string]model.AuditRowChange, len(dbResults))

	for _, result := range dbResults {
		rowID := strconv.FormatInt(result.RowID.Int64, 10)
		bulkInfo.Changes[rowID] = model.AuditRowChange{
			Payload:            result.Payload.String,
			AuditRowChangeBase: model.AuditRowChangeBase{EntityID: strconv.FormatInt(result.EntityID.Int64, 10), Action: result.Type.String},
		}
	}

	return &bulkInfo, nil
}

func (provider *Provider) GetAuditChangeInfo(ctx context.Context, changeIds []model.EntityID) (*model.AuditChangesInfo, error) {
	// many rows with different changes for one bulk
	var dbResults []GetAuditInfoResultBase

	query, args, err := sqlx.In(selectAuditChangesInfoQueryTemplate, changeIds)
	if err != nil {
		return nil, fmt.Errorf("failed to prepare select audit change info query: %s", err)
	}

	_, err = provider.withDriver(func(db *sqlx.DB) (interface{}, error) {
		return nil, db.SelectContext(ctx, &dbResults, db.Rebind(query), args...)
	})
	if err != nil {
		return nil, fmt.Errorf("failed to select audit changes: %s %s %s", err, query, args)
	}

	changesInfo := make(model.AuditChangesInfo, len(dbResults))

	for _, result := range dbResults {
		rowID := strconv.FormatInt(result.RowID.Int64, 10)
		changesInfo[rowID] = model.AuditChangeInfo{
			AuditRowChangeBase: model.AuditRowChangeBase{
				Action:   result.Type.String,
				EntityID: strconv.FormatInt(result.EntityID.Int64, 10),
			},
			AuditBulkInfo: model.AuditBulkInfoBase{
				ID:        strconv.FormatInt(result.ID.Int64, 10),
				Author:    result.Author.String,
				Comment:   result.Comment.String,
				Issue:     result.Issue.String,
				Timestamp: result.TS.Int64,
			},
		}
	}

	return &changesInfo, nil
}
