package mysql

import (
	"context"
	"fmt"
	"strconv"
	"time"

	"github.com/jmoiron/sqlx"

	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/filter"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/model"
)

const selectBlockedPhonesQueryTemplate = "SELECT blockid, phone, blocktype, blocktill, event_create, audit_bulk.timestamp AS event_create_ts, event_modify, audit_bulk2.timestamp AS event_modify_ts FROM blockedphones %s WHERE %s ORDER BY blockid LIMIT ?"
const selectBlockedPhonesPredicate = "blockid > ?"

const deleteBlockedPhonesQuery = "DELETE FROM blockedphones WHERE blockid IN (:delete)"

const updateBlockedPhonesQuery = "UPDATE blockedphones SET phone = :phone, blocktype = :blocktype, blocktill = :blocktill WHERE blockid = :blockid"

const insertBlockedPhonesQuery = "INSERT INTO blockedphones (blockid, phone, blocktype, blocktill) VALUES (:blockid, :phone, :blocktype, :blocktill)"

const blockedphonesTableName = "blockedphones"

type BlockedPhonesQueryParams struct {
	ID          uint64          `db:"blockid"`
	PhoneNumber string          `db:"phone"`
	BlockType   model.BlockType `db:"blocktype"`
	BlockUntil  time.Time       `db:"blocktill"`
}

var blockedPhonesFilterFields = map[string]string{
	model.BlockidAlias:   "blockid",
	model.BlocktypeAlias: "blocktype",
	model.BlocktillAlias: "blocktill",
	model.PhoneAlias:     "phone",
}

func (provider *Provider) GetBlockedPhonesCount(ctx context.Context, blockedPhonesFilter filter.Filter) (uint64, error) {
	return provider.GetCount(ctx, "blockedphones", blockedPhonesFilter, blockedPhonesFilterFields)
}

func (provider *Provider) GetBlockedPhones(
	ctx context.Context,
	fromID model.EntityID,
	limit uint64,
	filter filter.Filter) ([]*model.BlockedPhone, error) {
	return GetEntityInfo[model.BlockedPhone, GetBlockedPhonesResult](ctx, &limitSelectArgs{fromID, limit}, filter, DBEntitySpec{
		filterFields:        blockedPhonesFilterFields,
		selectPredicate:     selectBlockedPhonesPredicate,
		selectQueryTemplate: selectBlockedPhonesQueryTemplate,
		entityLogTitle:      "blockedphones",
		tableName:           blockedphonesTableName,
	}, provider)
}

func (provider *Provider) SetBlockedPhones(
	ctx context.Context,
	delete []model.EntityID,
	create []*model.BlockedPhone,
	update []*model.BlockedPhone,
	auditLogBulkParams *model.AuditLogBulkParams,
) error {
	batch := make(queryBatch, 0, 1+len(update)+len(create))

	/* NOTE: https://a.yandex-team.ru/arc_vcs/passport/infra/daemons/yasms_internal/internal/model/mysql/routes.go?rev=r8947415#L108 */
	if err := prepareDeleteQuery(&batch, deleteBlockedPhonesQuery, blockedphonesTableName, delete, Delete); err != nil {
		return fmt.Errorf("failed to prepare delete blockedphones query: %s", err)
	}

	if err := prepareBlockedPhonesQueries(&batch, updateBlockedPhonesQuery, update, Update); err != nil {
		return fmt.Errorf("failed to prepare update blockedphones query: %s", err)
	}

	if err := prepareBlockedPhonesQueries(&batch, insertBlockedPhonesQuery, create, Insert); err != nil {
		return fmt.Errorf("failed to prepare create blockedphones query: %s", err)
	}

	_, err := provider.withDriver(func(db *sqlx.DB) (interface{}, error) {
		return nil, batch.execSingleTransactionWithAuditLog(db, ctx, auditLogBulkParams)
	})
	return err
}

func prepareBlockedPhonesQueries(batch *queryBatch, queryTemplate string, update []*model.BlockedPhone, setType SetType) error {
	for _, blockedPhone := range update {
		var id uint64
		var insertedID *model.EntityID
		var err error
		if blockedPhone.ID == "" {
			insertedID = &blockedPhone.ID
		} else {
			if id, err = strconv.ParseUint(blockedPhone.ID, 10, 64); err != nil {
				return err
			}
		}

		query, args, err := prepareNamedQuery(
			queryTemplate,
			&BlockedPhonesQueryParams{
				ID:          id,
				PhoneNumber: blockedPhone.PhoneNumber,
				BlockType:   blockedPhone.BlockType,
				BlockUntil:  blockedPhone.BlockUntil,
			})
		if err != nil {
			return err
		}

		batch.append(&queryHolder{
			Query:        query,
			Args:         args,
			AffectedRows: 1,
			InsertedID:   insertedID,
			QueryMetaData: queryMetaData{
				ID:        []model.EntityID{blockedPhone.ID},
				TableName: blockedphonesTableName,
				Type:      setType,
				Payload:   blockedPhone.String(),
			},
		})
	}

	return nil
}
