package mysql

import (
	"context"
	"database/sql"
	"fmt"
	"strings"

	"github.com/jmoiron/sqlx"

	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/model"
)

type EntryType = int

const (
	alias     EntryType = 0
	alphaName EntryType = 1
)

const enumsQuery = `
SELECT ? AS type, aliase AS name, GROUP_CONCAT(DISTINCT fromname) AS entries FROM smsgates GROUP BY aliase
UNION
SELECT ? AS type, fromname AS name, GROUP_CONCAT(DISTINCT aliase) AS entries FROM smsgates GROUP BY fromname
`

func (provider *Provider) GetRouteEnums(ctx context.Context) (*model.Enums, error) {
	enums := &model.Enums{}

	rows, err := provider.withDriver(func(db *sqlx.DB) (interface{}, error) {
		return db.QueryContext(ctx, db.Rebind(enumsQuery), alias, alphaName)
	})
	if err != nil {
		return nil, fmt.Errorf("failed to select enums: %s", err)
	}

	enums.Aliases = map[string][]string{}
	enums.AlphaNames = map[string][]string{}

	for rows.(*sql.Rows).Next() {
		var entryType EntryType
		var name string
		var values string
		err = rows.(*sql.Rows).Scan(&entryType, &name, &values)
		if err != nil {
			return nil, fmt.Errorf("failed to scan enums entry: %s", err)
		}

		switch entryType {
		case alias:
			enums.Aliases[name] = strings.Split(values, ",")
		case alphaName:
			enums.AlphaNames[name] = strings.Split(values, ",")
		}
	}
	return enums, nil
}
